// ============================================================================
// Crash Survival Rating Pro v3.0
// Copyright (c) 2025 minMAX420 - All Rights Reserved
//
// Licensed under CC BY-NC-SA 4.0 (Creative Commons Attribution-NonCommercial-ShareAlike)
// https://creativecommons.org/licenses/by-nc-sa/4.0/
//
// PERSONAL USE ONLY - Commercial use is strictly prohibited.
// See LICENSE file for full terms.
// ============================================================================
//
// Real-Time G-Force Monitor with Injury Probability Estimation
// Based on NATO RTO-EN-HFM-113, NHTSA FMVSS 208, and biomechanical research
//
// ARCHITECTURE NOTE:
// This UI app performs real-time injury probability estimation based on:
// - Real accelerometer data (3-axis G-forces) via StreamsManager
// - Direction-specific injury modeling (lateral, longitudinal, vertical)
// - Duration effects and cumulative trauma
// - Body part specific injury tracking
//
// NOTE: All measurements are from actual vehicle sensors (accelerometers).
// Survival ratings are ESTIMATES based on crash research probability curves.

angular.module('beamng.apps').directive('crashSurvivalRating', ['$sce', function ($sce) {
  return {
    templateUrl: '/ui/modules/apps/crashsurvivalrating/app.html',
    replace: true,
    restrict: 'EA',
    link: function (scope, element, attrs) {
      console.log('crashSurvivalRating: UI App loading...');

      // ============================================
      // DYNAMIC SIZING - Scale based on container size
      // Baseline: 1200x620
      // ============================================
      var baselineWidth = 1200;
      var baselineHeight = 620;

      function updateDynamicSizing() {
        var container = element[0];
        var width = container.offsetWidth || baselineWidth;
        var height = container.offsetHeight || baselineHeight;

        // Calculate scale based on height (primary) with width as secondary factor
        var scaleH = height / baselineHeight;
        var scaleW = width / baselineWidth;
        // Use the smaller scale to ensure everything fits
        var scale = Math.min(scaleH, scaleW);

        // Clamp scale to reasonable bounds (0.5x to 2x)
        scale = Math.max(0.5, Math.min(2.0, scale));

        // Calculate scaled values (baseline values at 1200x620)
        var textXs = Math.round(12 * scale);
        var textSm = Math.round(14 * scale);
        var textMd = Math.round(18 * scale);
        var textLg = Math.round(24 * scale);
        var textXl = Math.round(32 * scale);
        var barHeight = Math.round(22 * scale);
        var gapSm = Math.round(6 * scale);
        var gapMd = Math.round(10 * scale);
        var padSm = Math.round(8 * scale);
        var padMd = Math.round(12 * scale);
        var borderWidth = Math.max(2, Math.round(3 * scale));
        var borderRadius = Math.round(6 * scale);

        // Apply to container element
        container.style.setProperty('--scale', scale);
        container.style.setProperty('--text-xs', textXs + 'px');
        container.style.setProperty('--text-sm', textSm + 'px');
        container.style.setProperty('--text-md', textMd + 'px');
        container.style.setProperty('--text-lg', textLg + 'px');
        container.style.setProperty('--text-xl', textXl + 'px');
        container.style.setProperty('--bar-height', barHeight + 'px');
        container.style.setProperty('--gap-sm', gapSm + 'px');
        container.style.setProperty('--gap-md', gapMd + 'px');
        container.style.setProperty('--pad-sm', padSm + 'px');
        container.style.setProperty('--pad-md', padMd + 'px');
        container.style.setProperty('--border-width', borderWidth + 'px');
        container.style.setProperty('--border-radius', borderRadius + 'px');

        if (scope.settings && scope.settings.debugMode) {
          console.log('crashSurvivalRating: Resized to', width + 'x' + height, 'scale:', scale.toFixed(2));
        }
      }

      // Initial sizing
      updateDynamicSizing();

      // Watch for resize using ResizeObserver
      var resizeObserver = null;
      if (typeof ResizeObserver !== 'undefined') {
        resizeObserver = new ResizeObserver(function() {
          updateDynamicSizing();
        });
        resizeObserver.observe(element[0]);
      } else {
        // Fallback: check size periodically
        var resizeInterval = setInterval(updateDynamicSizing, 500);
      }

      // Request electrics stream for acceleration data
      var streamsList = ['electrics'];
      if (typeof StreamsManager !== 'undefined') {
        StreamsManager.add(streamsList);
        console.log('crashSurvivalRating: Requested electrics stream via global');
      } else if (scope.StreamsManager) {
        scope.StreamsManager.add(streamsList);
        console.log('crashSurvivalRating: Requested electrics stream via scope');
      } else {
        console.error('crashSurvivalRating: StreamsManager not found!');
      }

      // Debug: Log when we receive data
      var updateCount = 0;

      // Initialize data object with default values
      scope.data = {
        currentGx: 0,
        currentGy: 0,
        currentGz: 0,
        currentGTotal: 0,
        maxGx: 0,
        maxGy: 0,
        maxGz: 0,
        maxGTotal: 0,
        survivalRating: 100,
        injuryLevel: 'Safe',
        highGDuration: 0,
        impactCount: 0,
        // Body part injuries (0-100)
        bodyParts: {
          head: 0,
          neck: 0,
          chest: 0,
          spine: 0,
          leftArm: 0,
          rightArm: 0,
          leftLeg: 0,
          rightLeg: 0
        },
        // Vehicle stats
        speed: 0,
        speedUnit: 'MPH',
        rpm: 0,
        gear: 'N',
        fuel: 0,
        throttle: 0,
        brake: 0,
        engineTemp: 0
      };

      // Settings
      scope.settings = {
        useMetric: false,
        showVehicleStats: true,
        showGForceBreakdown: true,
        showBodyPartIndicators: true,
        soundEnabled: false,
        debugMode: false
      };

      // Settings panel visibility
      scope.showSettings = false;

      // Heart Rate Monitor
      scope.heartRate = 72;
      var targetHeartRate = 72;
      var ecgCanvas = null;
      var ecgCtx = null;
      var ecgAnimationFrame = null;

      // Airtime Tracker
      scope.isAirborne = false;
      scope.currentAirtime = 0;
      scope.maxAirtime = 0;
      scope.rocketCrashed = false;
      var airtimeStart = 0;
      var wasAirborne = false;
      var crashTimeout = null;

      // ECG waveform - generates a single heartbeat shape
      function generateHeartbeatValue(phase, amplitude, noise) {
        // phase is 0-1 representing position within one heartbeat cycle
        var value = 0;

        // P wave (small bump at ~10-20%)
        if (phase >= 0.10 && phase < 0.20) {
          var p = (phase - 0.10) / 0.10;
          value = Math.sin(p * Math.PI) * amplitude * 0.15;
        }
        // Flat between P and QRS
        else if (phase >= 0.20 && phase < 0.30) {
          value = 0;
        }
        // Q dip (small dip before R)
        else if (phase >= 0.30 && phase < 0.33) {
          value = -amplitude * 0.1;
        }
        // R spike (big spike up!)
        else if (phase >= 0.33 && phase < 0.40) {
          var r = (phase - 0.33) / 0.07;
          value = Math.sin(r * Math.PI) * amplitude;
        }
        // S dip (dip after R)
        else if (phase >= 0.40 && phase < 0.45) {
          var s = (phase - 0.40) / 0.05;
          value = -amplitude * 0.25 * (1 - s);
        }
        // Flat between S and T
        else if (phase >= 0.45 && phase < 0.55) {
          value = 0;
        }
        // T wave (recovery bump)
        else if (phase >= 0.55 && phase < 0.70) {
          var t = (phase - 0.55) / 0.15;
          value = Math.sin(t * Math.PI) * amplitude * 0.2;
        }
        // Flat baseline
        else {
          value = 0;
        }

        // Add noise/jitter for erratic heartbeat during crashes
        if (noise > 0) {
          value += (Math.random() - 0.5) * noise * amplitude * 2;
        }

        return value;
      }

      // Monitor state variables
      var ecgDotX = 0;
      var ecgPhase = 0;
      var ecgTrail = []; // Array of {x, y} points for trail

      function initECG() {
        var canvasElements = element[0].querySelectorAll('#ecg-canvas');
        ecgCanvas = canvasElements.length > 0 ? canvasElements[0] : document.getElementById('ecg-canvas');

        if (!ecgCanvas) {
          setTimeout(initECG, 200);
          return;
        }

        ecgCtx = ecgCanvas.getContext('2d');
        var container = ecgCanvas.parentElement;
        ecgCanvas.width = Math.max(container.offsetWidth || 200, 100);
        ecgCanvas.height = Math.max(container.offsetHeight || 50, 40);

        ecgDotX = 0;
        ecgPhase = 0;
        ecgTrail = [];

        ecgCtx.fillStyle = '#000000';
        ecgCtx.fillRect(0, 0, ecgCanvas.width, ecgCanvas.height);

        animateECG();
      }


      function animateECG() {
        if (!ecgCanvas || !ecgCtx) {
          ecgAnimationFrame = requestAnimationFrame(animateECG);
          return;
        }

        var width = ecgCanvas.width;
        var height = ecgCanvas.height;
        if (width < 10 || height < 10) {
          ecgAnimationFrame = requestAnimationFrame(animateECG);
          return;
        }

        var centerY = height / 2;
        var injuryLevel = scope.data ? (scope.data.injuryLevel || 'Safe') : 'Safe';
        var isFatal = injuryLevel === 'Fatal';

        // Calculate heart rate based on injury AND airtime
        var noise = 0;
        var baseHeartRate = 72;

        // Base rate from injury level
        if (injuryLevel === 'Safe') { baseHeartRate = 72; noise = 0; }
        else if (injuryLevel === 'Minor Injury') { baseHeartRate = 100; noise = 0.1; }
        else if (injuryLevel === 'Moderate Injury') { baseHeartRate = 130; noise = 0.2; }
        else if (injuryLevel === 'Severe Injury') { baseHeartRate = 165; noise = 0.35; }
        else if (injuryLevel === 'Critical') { baseHeartRate = 200; noise = 0.5; }
        else if (injuryLevel === 'Fatal') { baseHeartRate = 0; noise = 0; }

        // Airtime excitement boost! (adrenaline rush during jumps)
        var airtimeBoost = 0;
        if (scope.isAirborne && scope.currentAirtime > 0) {
          // Heart rate increases with airtime duration
          // +20 BPM at 1 sec, +40 at 2 sec, +60 at 3+ sec
          airtimeBoost = Math.min(60, scope.currentAirtime * 20);
          // Add some excitement noise
          noise = Math.max(noise, 0.15 + scope.currentAirtime * 0.1);
        }
        // Landing spike - brief heart rate spike when crashing back down
        if (scope.rocketCrashed) {
          airtimeBoost = 40; // Spike on landing
          noise = Math.max(noise, 0.3);
        }

        targetHeartRate = Math.min(220, baseHeartRate + airtimeBoost); // Cap at 220 BPM
        if (isFatal) targetHeartRate = 0;

        scope.heartRate = Math.round(scope.heartRate + (targetHeartRate - scope.heartRate) * 0.05);
        if (scope.heartRate < 1 && isFatal) scope.heartRate = 0;

        // Speed and amplitude
        var pixelsPerBeat = 80;
        var beatsPerSecond = Math.max(scope.heartRate, 10) / 60;
        var pixelsPerFrame = (beatsPerSecond * pixelsPerBeat) / 60;
        var amplitude = height * 0.35;

        // Calculate Y value
        var ecgValue;
        if (isFatal) {
          ecgValue = (Math.random() - 0.5) * 2;
        } else {
          ecgPhase += pixelsPerFrame / pixelsPerBeat;
          if (ecgPhase >= 1) ecgPhase -= 1;
          ecgValue = generateHeartbeatValue(ecgPhase, amplitude, noise);
        }
        var currentY = centerY - ecgValue;

        // Add point to trail
        ecgTrail.push({ x: ecgDotX, y: currentY });

        // Keep trail limited
        var maxTrailLength = 120;
        while (ecgTrail.length > maxTrailLength) {
          ecgTrail.shift();
        }

        // Clear canvas
        ecgCtx.fillStyle = '#000000';
        ecgCtx.fillRect(0, 0, width, height);

        // Draw faint grid
        ecgCtx.strokeStyle = 'rgba(51, 255, 102, 0.08)';
        ecgCtx.lineWidth = 1;
        for (var gx = 0; gx < width; gx += 20) {
          ecgCtx.beginPath();
          ecgCtx.moveTo(gx, 0);
          ecgCtx.lineTo(gx, height);
          ecgCtx.stroke();
        }

        // Draw trail with fade effect (green)
        var lineColor = isFatal ? '#FF1133' : '#33FF66';
        for (var i = 1; i < ecgTrail.length; i++) {
          var alpha = i / ecgTrail.length;
          ecgCtx.beginPath();
          ecgCtx.strokeStyle = isFatal ?
            'rgba(255, 17, 51, ' + (alpha * 0.9) + ')' :
            'rgba(51, 255, 102, ' + (alpha * 0.9) + ')';
          ecgCtx.lineWidth = 2;
          ecgCtx.moveTo(ecgTrail[i-1].x, ecgTrail[i-1].y);
          ecgCtx.lineTo(ecgTrail[i].x, ecgTrail[i].y);
          ecgCtx.stroke();
        }

        // Draw glowing dot (GREEN)
        var gradient = ecgCtx.createRadialGradient(ecgDotX, currentY, 0, ecgDotX, currentY, 15);
        gradient.addColorStop(0, isFatal ? 'rgba(255, 50, 50, 1)' : 'rgba(100, 255, 150, 1)');
        gradient.addColorStop(0.3, isFatal ? 'rgba(255, 30, 30, 0.6)' : 'rgba(51, 255, 102, 0.6)');
        gradient.addColorStop(1, 'rgba(0, 0, 0, 0)');
        ecgCtx.beginPath();
        ecgCtx.fillStyle = gradient;
        ecgCtx.arc(ecgDotX, currentY, 15, 0, Math.PI * 2);
        ecgCtx.fill();

        // Bright center dot
        ecgCtx.beginPath();
        ecgCtx.fillStyle = isFatal ? '#FF6666' : '#AAFFCC';
        ecgCtx.arc(ecgDotX, currentY, 4, 0, Math.PI * 2);
        ecgCtx.fill();

        // White core
        ecgCtx.beginPath();
        ecgCtx.fillStyle = '#FFFFFF';
        ecgCtx.arc(ecgDotX, currentY, 2, 0, Math.PI * 2);
        ecgCtx.fill();

        // Advance dot position (left to right)
        ecgDotX += pixelsPerFrame;
        if (ecgDotX >= width) {
          ecgDotX = 0;
          ecgTrail = [];
        }

        ecgAnimationFrame = requestAnimationFrame(animateECG);
      }

      // Get airtime display class
      scope.getAirtimeClass = function() {
        if (scope.rocketCrashed) return 'crashed';
        if (!scope.isAirborne) return '';
        if (scope.currentAirtime >= 3) return 'epic';
        return 'airborne';
      };

      // Get rocket animation state
      scope.getRocketState = function() {
        if (scope.rocketCrashed) return 'crashed';
        if (scope.isAirborne) {
          if (scope.currentAirtime >= 3) return 'airborne epic';
          return 'airborne';
        }
        return 'grounded';
      };

      // Get heart rate display class
      scope.getHeartRateClass = function() {
        if (scope.heartRate === 0) return 'flatline';
        if (scope.heartRate >= 180) return 'critical';
        if (scope.heartRate >= 140) return 'danger';
        if (scope.heartRate >= 100) return 'elevated';
        return '';
      };


      // Initialize monitors after DOM is ready
      setTimeout(function() {
        initECG();
        // Backup retry in case first attempt fails
        setTimeout(function() {
          if (!ecgCanvas || !ecgCtx) {
            initECG();
          }
        }, 500);
      }, 300);

      // Character sayings system with typewriter effect
      scope.currentSaying = "";
      scope.displayedSaying = $sce.trustAsHtml("");
      scope.isTyping = false;
      var fullSayingText = "Ready to roll!";
      var typewriterIndex = 0;
      var typewriterInterval = null;
      var typewriterSpeed = 35; // ms per character

      var sayings = {
        safe: [
          // Smooth driving comments
          "Nice and smooth, keep it up!",
          "This is how grandma drives. Respect.",
          "Boring... but my spine thanks you.",
          "Safety first! My insurance agent approves.",
          "Yawn... wake me when we crash.",
          "Are we there yet?",
          "I could do this all day.",
          "Smooth operator over here.",
          "Look ma, no hands! ...kidding.",
          "Professional driver on a closed course.",
          "This is fine. Everything is fine.",
          "Just vibing with my seatbelt on.",
          "My grandma drives faster than this.",
          "Scenic route, right?",
          "Are we saving fuel or what?",
          "Sunday driver energy.",
          // Speed/cruising comments
          "Cruising altitude achieved.",
          "Ah yes, the speed limit. A suggestion.",
          "Wind in my hair... if I had any.",
          "Living the dream right here.",
          "This is my jam! *air drums*",
          "Smooth like butter on a hot pan.",
          "I could get used to this.",
          "Zero complaints from the passenger.",
          "My neck thanks you for this.",
          "Finally, someone who can drive!",
          "Keep calm and drive on.",
          "This is... actually nice?",
          "No whiplash today, folks!",
          "A+ driving, would ride again.",
          "Insurance rates staying LOW today.",
          "My spine is intact. Novel concept.",
          // Sarcastic safe comments
          "Wow, using turn signals and everything!",
          "Is this what responsible driving feels like?",
          "My therapist would be proud.",
          "Look at us, obeying traffic laws!",
          "Defensive driving? In THIS economy?",
          "Someone took driver's ed seriously.",
          "Plot twist: we arrive alive!",
          "No near-death experiences yet. Weird.",
          "Actually following the road? Revolutionary!",
          "My heart rate is... normal? Strange.",
          // Bored safe comments
          "I've seen more action in a parking lot.",
          "This is nice but... floor it?",
          "Safety is cool but have you tried SPEED?",
          "My adrenaline glands are confused.",
          "Is the speedometer broken or...?",
          "Wake me when we hit triple digits.",
          "The fast lane is over THERE.",
          "I didn't wear my racing undies for THIS."
        ],
        minor: [
          // Small bump reactions
          "Whoa! Easy there, Speed Racer!",
          "My coffee just spilled!",
          "That's gonna leave a mark... on my pride.",
          "Did we hit a pothole or a small car?",
          "Ow! I bit my tongue!",
          "Was that a speed bump or a turtle?",
          "My chiropractor thanks you for the business.",
          "Felt that one in my fillings!",
          "Hey! I'm trying to text here!",
          "Was that a curb or my spine?",
          "Careful! I just had lunch!",
          "My neck did NOT need that.",
          "Pretty sure that was on purpose.",
          "Do you even HAVE a license?!",
          "That pothole had a family!",
          "Warn me next time!",
          // Startled reactions
          "Whoa there, cowboy!",
          "A little warning would be nice!",
          "My drink! MY DRINK!",
          "That's not a shortcut, that's a wall!",
          "Bumpy road or bad driver? Discuss.",
          "I almost dropped my phone!",
          "Seatbelt earning its keep today!",
          "Was that a curb check? Failed.",
          "Minor turbulence... in a CAR.",
          "Road surface: 2/10, would not recommend.",
          "Did you MEAN to hit that?",
          "My spine just filed a complaint.",
          "That pothole has a name now. Steve.",
          "RIP to my tailbone.",
          "Ow, my everything!",
          "The suspension is crying.",
          // Annoyed reactions
          "Really? REALLY?",
          "I will remember this.",
          "Adding that to your tab.",
          "My Yelp review will reflect this.",
          "One star. Would not ride again.",
          "I've had smoother elevator rides.",
          "Were you aiming for that?",
          "This isn't rally racing, chief.",
          "Mirrors exist for a reason.",
          "That was definitely avoidable.",
          "I saw that coming. Why didn't you?",
          "Taking notes for my insurance claim.",
          "Interesting driving technique...",
          "Bold strategy, let's see if it pays off.",
          "Points for creativity, I guess?"
        ],
        moderate: [
          // Impact reactions
          "OUCH! That actually hurt!",
          "I think I saw my life flash by!",
          "Who put that wall there?!",
          "My neck is NOT supposed to bend that way!",
          "Pretty sure that voided the warranty!",
          "Is it too late to buy insurance?",
          "I should have worn the brown pants!",
          "That's coming out of YOUR deposit!",
          "I'd like to speak to your driving instructor!",
          "Was that the brake or the accelerator?!",
          "My lawyer will hear about this!",
          "I didn't sign up for this!",
          "That tree came out of NOWHERE!",
          "Define 'controlled landing'...",
          "I think I lost a filling!",
          "Pretty sure my organs shifted!",
          // Damage awareness
          "I heard something crack. Was that the car or me?",
          "That's gonna buff out... right? RIGHT?!",
          "The car is making new noises now.",
          "Something fell off. Don't look back.",
          "I think that was important. The thing that flew off.",
          "We're definitely not getting the deposit back.",
          "How many airbags does this thing have?",
          "Crumple zones doing their job, I see.",
          "That's one way to test the bumper.",
          "New modification: unplanned body work.",
          "Panel gaps? We have panel CANYONS now.",
          "Structural integrity: questionable.",
          "The frame is now abstract art.",
          "I think the wheel is supposed to be UNDER us.",
          "That grinding noise isn't good, right?",
          "Smoke is never a good sign.",
          // Panic reactions
          "BRAKE BRAKE BRAKE BRAKE!",
          "WRONG PEDAL! WRONG PEDAL!",
          "TREES DON'T MOVE!",
          "WALLS ARE SOLID! WHO KNEW?!",
          "THAT'S NOT A ROAD!",
          "PHYSICS! REMEMBER PHYSICS!",
          "GRAVITY IS NOT YOUR FRIEND!",
          "STEERING! USE THE STEERING!",
          "THE OTHER LEFT!",
          "I SAID SLOW DOWN!",
          "THIS ISN'T FORZA!",
          "RULES OF THE ROAD! REMEMBER THEM!",
          // Vehicle impact specific
          "DID WE JUST HIT THAT CAR?!",
          "THE BUMPER! WHERE'S THE BUMPER?!",
          "THERE GOES THE MIRROR!",
          "THE HOOD IS BLOCKING THE VIEW!",
          "SPARKS! WHY ARE THERE SPARKS?!",
          "WE'RE DRAGGING SOMETHING!",
          "THE TIRE JUST EXPLODED!",
          "SMOKE FROM THE ENGINE! SMOKE!",
          "THE STEERING FEELS WRONG!",
          "WE'RE LOSING CONTROL!",
          "T-BONE! WE JUST GOT T-BONED!",
          "REAR-ENDED! WHO REAR-ENDED US?!"
        ],
        severe: [
          // Extreme panic
          "AAAHHH! MAKE IT STOP!",
          "I regret everything!",
          "Tell my family I love them!",
          "THIS IS NOT A DRILL!",
          "Why did I get in this car?!",
          "I see the light... wait, that's the airbag!",
          "Medic! MEDIC!",
          "That's not how physics should work!",
          "I WANT OFF THIS RIDE!",
          "MAYDAY MAYDAY MAYDAY!",
          "This was a MISTAKE!",
          "I KNEW I should have walked!",
          "Calling my mom... if I survive!",
          "Is this how it ends?!",
          "NOT THE BEES! Wait wrong movie.",
          "I'm gonna be sick!",
          // Body damage awareness
          "I can taste colors now!",
          "My spine is a slinky!",
          "Legs are overrated anyway!",
          "That's not where my arm goes!",
          "I can see my own back now!",
          "My skeleton wants a divorce!",
          "Bones aren't supposed to BEND!",
          "I've been folded like laundry!",
          "My joints are now UNIVERSAL joints!",
          "I've achieved pretzel form!",
          "Flexibility level: EXTREME!",
          "I'm inside out! Wait, no, I'm fine. No wait-",
          "The human body wasn't designed for this!",
          "New yoga pose: THE CAR CRASH!",
          "I can lick my own elbow now! Wait, that's bad.",
          "My chiropractor just felt a disturbance.",
          // Desperate pleas
          "I'LL BE GOOD! I PROMISE!",
          "I'LL NEVER SPEED AGAIN!",
          "TAKE THE CAR, NOT MY LIFE!",
          "I'M NOT READY!",
          "WHY DIDN'T I STAY HOME?!",
          "I'LL START GOING TO THE GYM!",
          "I'LL CALL MY MOTHER MORE!",
          "I'LL RECYCLE! I SWEAR!",
          "PLEASE LET THIS BE A DREAM!",
          "I TAKE BACK EVERYTHING BAD I'VE SAID!",
          "I'M A GOOD PERSON! MOSTLY!",
          "KARMA? IS THAT YOU?!",
          // Crash-specific body damage reactions
          "THE AIRBAG PUNCHED ME IN THE FACE!",
          "I THINK I LEFT MY STOMACH BACK THERE!",
          "WE'RE ROLLING! WE'RE ROLLING!",
          "THE ROOF IS NOW THE FLOOR!",
          "WHICH WAY IS UP?!",
          "STOP SPINNING! PLEASE STOP!",
          "I CAN HEAR THE METAL CRUNCHING!",
          "THE DOOR IS IN MY LAP NOW!",
          "GLASS EVERYWHERE! GLASS EVERYWHERE!",
          "WE'RE STILL MOVING! WHY ARE WE STILL MOVING?!",
          "THE ENGINE IS IN THE CABIN!",
          "I CAN SEE OUTSIDE THROUGH THE FLOOR!"
        ],
        critical: [
          // Incoherent panic
          "*incoherent screaming*",
          "I'M TOO YOUNG TO DIE!",
          "WITNESS ME!",
          "This is fine. Everything is fine.",
          "My skeleton is now a jigsaw puzzle!",
          "I have so many regrets!",
          "GAME OVER MAN, GAME OVER!",
          "Why didn't I take the bus?!",
          "DELETE MY BROWSER HISTORY!",
          "Tell my wife... she was right about everything!",
          "LEEEEROYYY JENKINSSSS!",
          "I REGRET NOTHING! ...okay some things!",
          "WASTED",
          "This is NOT what I meant by living on the edge!",
          "I see dead people... wait, that's me!",
          "Rolling... rolling... rolling...",
          "At least it's not boring anymore!",
          // Gaming/meme references
          "FATALITY!",
          "K.O.!",
          "FINISH HIM! Wait, no, don't!",
          "Wrecked. Absolutely wrecked.",
          "That's gonna leave a mark!",
          "X_X",
          "Critical hit! It's super effective!",
          "YOU DIED... almost!",
          "MISSION FAILED - We'll get 'em next time.",
          "Loading last checkpoint...",
          "Achievement Unlocked: Still Alive (Barely)",
          "Difficulty: NIGHTMARE",
          "Survival chance: lol",
          "Health bar: |________|",
          "Status: It's complicated.",
          "Error 404: Survival not found.",
          // Existential crisis
          "Is this real? Am I real?",
          "I've made a series of poor choices.",
          "Life flashing by... mostly regrets.",
          "So this is how it ends. In a Pigeon.",
          "I should have been a dentist.",
          "Mom was right about everything.",
          "My horoscope did NOT mention this.",
          "The fortune cookie LIED.",
          "I knew I shouldn't have left the house.",
          "Mercury must be in retrograde.",
          "I blame society.",
          "Plot twist: there is no plot, only pain.",
          // Acceptance
          "Well, it's been real.",
          "So this is how my story ends.",
          "At least I went out spectacularly.",
          "They'll tell tales of this crash.",
          "Legend status: ACHIEVED (posthumously)",
          "Going out with a bang! Literally!",
          "Making memories! Traumatic ones!",
          "This'll make a great story. If I survive.",
          "Future me is gonna feel this.",
          "Current me is already feeling this.",
          "Worth it? Jury's still out.",
          // Cursing outbursts (colored) - crash context: violent impact reactions
          "HOLY <span class='curse'>SH*T</span>! HOLY <span class='curse'>SH*T</span>!",
          "WHAT THE <span class='curse'>F***</span>?!",
          "OH <span class='curse'>F***</span> OH <span class='curse'>F***</span> OH <span class='curse'>F***</span>!",
          "SON OF A <span class='curse'>B****</span>!",
          "<span class='curse'>JESUS F***ING CHRIST</span>!",
          "MOTHER OF <span class='curse'>GOD</span>!",
          "<span class='curse'>GOD DAMN</span> IT ALL!",
          "ARE YOU <span class='curse'>F***ING</span> KIDDING ME?!",
          "WHAT THE ACTUAL <span class='curse'>F***</span>!",
          "OH <span class='curse'>HELL</span> NO!",
          "<span class='curse'>F***</span> THIS! <span class='curse'>F***</span> THAT! <span class='curse'>F***</span> EVERYTHING!",
          "<span class='curse'>SH*T SH*T SH*T SH*T</span>!",
          "<span class='curse'>BULL$#!T</span>!",
          "<span class='curse'>A$$HOLE</span>! THE TREE IS AN <span class='curse'>A$$HOLE</span>!",
          "WHO THE <span class='curse'>F***</span> PUT THAT THERE?!",
          "KISS MY <span class='curse'>A$$</span> PHYSICS!",
          "<span class='curse'>F***</span> ME SIDEWAYS!",
          "<span class='curse'>DAMN</span> IT TO <span class='curse'>HELL</span>!",
          "<span class='curse'>BLOODY HELL</span>!",
          "SWEET MOTHER OF <span class='curse'>F***</span>!",
          "<span class='curse'>@#$%&*</span>!!!",
          "<span class='curse'>$#!T</span> SANDWICH!",
          "PIECE OF <span class='curse'>$#!T</span> CAR!",
          "<span class='curse'>F***ING</span> GREAT! JUST <span class='curse'>F***ING</span> GREAT!",
          "OH FOR <span class='curse'>F***'S</span> SAKE!",
          // More crash-specific curses
          "THE <span class='curse'>F***ING</span> WALL CAME OUT OF NOWHERE!",
          "MY <span class='curse'>DAMN</span> NECK! MY <span class='curse'>DAMN</span> SPINE!",
          "<span class='curse'>SCREW</span> THIS CAR! <span class='curse'>SCREW</span> THIS ROAD!",
          "I'M GONNA <span class='curse'>F***ING</span> DIE IN A PIGEON!",
          "WHO DESIGNED THIS <span class='curse'>$#!T</span>?!",
          "THIS IS <span class='curse'>BULL$#!T</span>! TOTAL <span class='curse'>BULL$#!T</span>!",
          "<span class='curse'>AAAARGH</span>! MY <span class='curse'>F***ING</span> EVERYTHING!"
        ],
        fatal: [
          // Silent/death
          "...",
          "*flatline noises*",
          "Well... that happened.",
          "F",
          "Respawning in 3... 2... 1...",
          "Should've stayed in bed today.",
          "At least I didn't feel that last one.",
          "Achievement Unlocked: Speed Bump",
          "GG no re",
          "You Died",
          "Continue? 9... 8... 7...",
          "Insert coin to continue",
          "Task failed successfully.",
          "Return to main menu?",
          "*Windows shutdown sound*",
          "Directed by Robert B. Weide",
          "To be continued...",
          "He lived as he died: terribly.",
          // Game over references
          "GAME OVER",
          "Thanks for playing!",
          "Better luck next time!",
          "Your princess is in another castle.",
          "WASTED",
          "Busted.",
          "Try again? [Y/N]",
          "Save file corrupted.",
          "Ragdoll physics: ACTIVATED",
          "Respawn point: Heaven",
          "Loading screen of eternity...",
          "Press F to pay respects.",
          "Scoreboard: 0-1 (Tree wins)",
          "New high score! (In damage)",
          "Speedrun complete. Wrong category.",
          "Any% crash strats successful.",
          // Dark humor
          "I lived as I died: poorly.",
          "Subscribed to the forever sleep.",
          "Room temperature challenge: ACCEPTED",
          "Now accepting flowers.",
          "Permanently AFK.",
          "Logged out of life.",
          "Connection terminated.",
          "Account deleted.",
          "Uninstalled from existence.",
          "Sent to the shadow realm.",
          "Currently experiencing technical difficulties.",
          "Service unavailable. Error: DEAD",
          "Gone to the big junkyard in the sky.",
          "Now one with the pavement.",
          "Became a permanent roadside attraction.",
          "Achieved terminal velocity. And then terminal.",
          // Acceptance/aftermath
          "It's quiet now.",
          "*peaceful silence*",
          "The simulation ended.",
          "Screen fading to black...",
          "Credits rolling...",
          "Executive Producer: Bad Decisions",
          "Written by: Gravity",
          "Sponsored by: Poor Life Choices",
          "In memory of: My driving record",
          "No cars were unharmed in this production.",
          "Stunt driver: ME (unfortunately)",
          "This crash was performed by a professional. Results not typical. Or survivable.",
          // Wingdings/Symbol gibberish (corrupted last words) - colored
          "<span class='symbol'>☠️💀☠️💀☠️</span>",
          "<span class='dead'>✞ R.I.P ✞</span>",
          "<span class='symbol'>◼◼◼◼◼◼◼◼</span>",
          "<span class='symbol'>▓▓▓</span> <span class='dead'>SIGNAL LOST</span> <span class='symbol'>▓▓▓</span>",
          "<span class='curse'>⚠️ FATAL ERROR ⚠️</span>",
          "<span class='symbol'>☢️ BIOHAZARD ☢️</span>",
          "<span class='symbol'>⬛⬛⬛</span> <span class='dead'>NO CARRIER</span> <span class='symbol'>⬛⬛⬛</span>",
          "<span class='symbol'>█▀█ █▄█ ▀█▀</span>",
          "<span class='symbol'>╔═══════════╗</span> <span class='dead'>DECEASED</span> <span class='symbol'>╚═══════════╝</span>",
          "<span class='symbol'>【ＤＥＡＤ】</span>",
          "<span class='symbol'>乇乂丅尺卂 ᗪ乇卂ᗪ</span>",
          "<span class='symbol'>▄︻̷̿┻̿═━一</span> <span class='curse'>💀</span>",
          "<span class='curse'>✖_✖</span>",
          "<span class='curse'>( ✖ _ ✖ )</span>",
          "<span class='symbol'>(╯°□°)╯︵ ┻━┻</span> ...<span class='curse'>💀</span>",
          "<span class='symbol'>¯\\_(ツ)_/¯</span> <span class='dead'>*dies*</span>",
          "<span class='symbol'>ಠ_ಠ</span> → <span class='curse'>✖_✖</span>",
          "<span class='symbol'>( ͡° ͜ʖ ͡°)</span> → <span class='curse'>( ͡✖ ͜ʖ ͡✖)</span>",
          "<span class='symbol'>░░░░░░░░░░░░</span>",
          "<span class='symbol'>◉_◉</span> → <span class='curse'>✗_✗</span>",
          "<span class='symbol'>■■■</span> <span class='curse'>CORRUPTED</span> <span class='symbol'>■■■</span>",
          "<span class='symbol'>♪~ ᕕ(ᐛ)ᕗ</span> → <span class='curse'>💀</span>",
          "<span class='dead'>01000100 01000101 01000001 01000100</span>",
          "<span class='dead'>//LIFE_PROCESS_TERMINATED//</span>",
          "<span class='curse'>ERR_CONNECTION_RESET_BY_DEATH</span>",
          "<span class='dead'>sudo rm -rf /life/*</span>",
          "<span class='dead'>git commit -m 'final'</span>",
          "<span class='dead'>return null; // forever</span>",
          "<span class='dead'>} // end of story</span>",
          "<span class='dead'>&lt;!-- user has left the chat --&gt;</span>",
          "<span class='curse'>404: LIFE NOT FOUND</span>",
          "<span class='curse'>kernel panic: not syncing</span>",
          "<span class='curse'>segmentation fault (core dumped)</span>"
        ],
        highg: [
          // G-force descriptions
          "G-FORCES... CRUSHING... ME...",
          "Can't... breathe... too many... Gs!",
          "My face is melting off!",
          "Fighter pilots train for this. I didn't!",
          "Blood leaving brain... getting dizzy...",
          "Is this what astronauts feel?!",
          "I can see through time!",
          "My cheeks are in the back seat!",
          "Eyeballs... retreating... into skull...",
          "Lungs... deflating...",
          "This must be what a pancake feels like!",
          "I'm becoming ONE with the seat!",
          // Hard cornering
          "LEAN INTO IT! LEAN!",
          "Centrifugal force is NOT a myth!",
          "I'm stuck to the door!",
          "Lateral G's are my nemesis!",
          "Why is the world sideways?!",
          "I can feel my brain shifting!",
          "My organs are rearranging!",
          "Inertia is a harsh mistress!",
          "The turn never ends!",
          "Still turning... still turning...",
          "Is this a corner or a black hole?!",
          "My blood is all on one side now!",
          // Hard acceleration/braking
          "WHIPLASH INCOMING!",
          "My neck! My poor neck!",
          "Brake check from HELL!",
          "0-100 in my neck hurts!",
          "Deceleration is just backwards acceleration AND IT HURTS!",
          "I can see my future... and it's pain!",
          "Momentum: 1, Me: 0",
          "Newton's laws are attacking me!",
          "Equal and opposite reaction... to my spine!",
          "That's aggressive acceleration!",
          "The seatbelt is cutting me in half!",
          "My internal organs switched places!",
          // Airplane/space references
          "Requesting permission to vomit!",
          "Houston, we have a problem!",
          "Ground control to Major Tom...",
          "This is your captain speaking: AHHH!",
          "Turbulence level: EXTREME!",
          "Preparing for re-entry!",
          "Escape velocity achieved!",
          "Orbital mechanics, but make it painful!",
          "Space would be more comfortable!",
          "At least astronauts get training!",
          // Physical descriptions
          "My face is doing weird things!",
          "Skin... peeling back...",
          "Jowls have achieved flight!",
          "My eyeballs want out!",
          "Spine compression engaged!",
          "Blood pooling in weird places!",
          "Vision... tunneling...",
          "Hearing my heartbeat in my ears!",
          "Consciousness... fading...",
          "Gray out in 3... 2...",
          "Red mist forming...",
          "Can't... lift... arms..."
        ],
        airborne: [
          // Excitement/thrill
          "WOOOHOOO! WE'RE FLYING!",
          "I BELIEVE I CAN FLY!",
          "TO INFINITY AND BEYOND!",
          "THIS IS AMAZING!",
          "YEEEEHAWWW!",
          "I'M A BIRD!",
          "WE GOT AIR!",
          "SICK JUMP BRO!",
          "Maximum air achieved!",
          "We're airborne, baby!",
          "Look ma, no ground!",
          "Defying gravity since today!",
          "Wheels? Who needs 'em!",
          "We're in the air! ON PURPOSE!",
          "Catching some serious hang time!",
          // Fear of landing
          "What goes up must... oh no.",
          "Landing is future me's problem!",
          "Gravity is about to remember us...",
          "The ground is getting closer. FAST.",
          "I regret this already!",
          "WHO THOUGHT THIS WAS A GOOD IDEA?!",
          "We're gonna feel that landing!",
          "Prepare for impact!",
          "BRACE BRACE BRACE!",
          "This seemed cooler in my head!",
          "The car wasn't designed for this!",
          "Where's the runway?!",
          "Mayday! We're coming in hot!",
          "Landing gear? We have landing gear?!",
          "The suspension is gonna HATE us!",
          // Weightlessness sensations
          "My stomach is floating!",
          "Zero-G! Zero-G!",
          "Is this what freefall feels like?!",
          "Weightless! I'm weightless!",
          "My organs are confused!",
          "Everything is floating in here!",
          "Space program, here I come!",
          "Houston, we have liftoff!",
          "One small step for car, one giant leap for me!",
          "Achieving orbit!",
          "Escape velocity: ACHIEVED!",
          "NASA called, they want their astronaut back!",
          // Time awareness
          "Still in the air... still in the air...",
          "How long have we been up here?!",
          "Time slows down when you're airborne!",
          "Still flying... this is fine...",
          "We've been up here a while now...",
          "New record! Maybe! Probably!",
          "This is lasting longer than expected!",
          "Ground? Never heard of her.",
          // General airtime comments
          "WHEEEEE!",
          "Up, up, and away!",
          "Flying without a license!",
          "The birds are jealous!",
          "Air time! Air time!",
          "Stunt driver of the year!",
          "X Games here I come!",
          "Tony Hawk wishes he had this air!",
          "Dukes of Hazzard vibes!",
          "General Lee, eat your heart out!",
          "Movie stunt energy!",
          "Sponsored by Red Bull!",
          "SEND IT!"
        ],
        idle: [
          // Waiting/bored
          "Hey, you gonna drive or what?",
          "I'm just gonna stand here looking cool.",
          "*elevator music plays*",
          "Waiting for something exciting to happen...",
          "The suspense is killing me. Slowly.",
          "Tick tock, let's rock!",
          "*whistling innocently*",
          "I could've taken an Uber.",
          "So... nice weather we're having.",
          "Is this car even on?",
          "Parking simulator champion!",
          "Wake me when something happens.",
          "Rent's due while we wait.",
          "My other car is also not moving.",
          // Impatient
          "Any day now...",
          "I could walk faster than this.",
          "The engine is getting cold. And so am I.",
          "Did we break down?",
          "Is this part of the experience?",
          "Loading... loading... still loading...",
          "The scenic route of sitting still.",
          "Taking the concept of 'park' literally.",
          "Parking: Expert Mode",
          "Stationary AF.",
          "Are we there yet? We haven't moved.",
          "This isn't how racing works.",
          "I've seen faster glaciers.",
          "The handbrake misses you too.",
          "Gas pedal. Right one. Try it.",
          "The accelerator won't bite.",
          // Sarcastic
          "Bold choice. Doing nothing.",
          "This is riveting. Truly.",
          "What an adventure. Sitting.",
          "Living on the edge. Of the parking spot.",
          "Thrill ride of the century right here.",
          "Edge of my seat stuff. Really.",
          "The speedometer is judging you.",
          "Even the GPS is confused.",
          "Recalculating... still recalculating...",
          "Are we playing Parking Simulator?",
          "Speed: 0. Fun: Also 0.",
          "Zero to zero in no time flat!",
          // Environmental observations
          "Nice day for a drive. Just saying.",
          "That's a nice tree. Been staring at it.",
          "I've counted 47 things we could hit.",
          "The road is RIGHT THERE.",
          "Other cars seem to be moving. Weird flex.",
          "I spy with my little eye... nothing happening.",
          "People are walking past us. On foot.",
          "A snail just passed us.",
          "The world is moving. We are not.",
          "Traffic report: We ARE the traffic.",
          "Current activity: Existing. Barely.",
          "Fuel consumption: Zero. Adventure: Also zero."
        ]
      };

      var lastSayingState = '';
      var sayingCooldown = 0;
      var sayingMinInterval = 3; // Minimum seconds between saying changes
      var sayingChance = 0.35; // 35% chance to show a saying on crash
      var crashStates = ['minor', 'moderate', 'severe', 'critical', 'fatal', 'highg', 'airborne'];

      function getRandomSaying(state) {
        var stateArray = sayings[state] || sayings.safe;
        return stateArray[Math.floor(Math.random() * stateArray.length)];
      }

      function startTypewriter(text) {
        // Clear any existing typewriter
        if (typewriterInterval) {
          clearInterval(typewriterInterval);
        }

        fullSayingText = text;
        typewriterIndex = 0;
        scope.displayedSaying = $sce.trustAsHtml("");
        scope.isTyping = true;

        // For HTML content, we need to handle tags properly
        // Extract plain text for typewriter effect, then apply HTML at end
        var plainText = text.replace(/<[^>]*>/g, '');
        var htmlText = text;

        typewriterInterval = setInterval(function() {
          if (typewriterIndex < plainText.length) {
            // Show plain text during typing
            scope.displayedSaying = $sce.trustAsHtml(plainText.substring(0, typewriterIndex + 1));
            typewriterIndex++;
            scope.$apply();
          } else {
            // Show full HTML when done
            scope.displayedSaying = $sce.trustAsHtml(htmlText);
            clearInterval(typewriterInterval);
            typewriterInterval = null;
            scope.isTyping = false;
            scope.$apply();
          }
        }, typewriterSpeed);
      }

      function updateSaying(currentState) {
        // Only update if state changed AND cooldown expired
        if (sayingCooldown > 0) return;

        // If state changed
        if (currentState !== lastSayingState) {
          // For crash states, apply random chance
          var isCrashState = crashStates.indexOf(currentState) !== -1;
          var wasCalm = lastSayingState === 'safe' || lastSayingState === 'idle' || lastSayingState === '';

          // Show saying if: entering crash state from calm OR random chance hits OR not a crash state
          var shouldShow = !isCrashState || (wasCalm && isCrashState) || Math.random() < sayingChance;

          if (shouldShow) {
            var newSaying = getRandomSaying(currentState);
            startTypewriter(newSaying);
            sayingCooldown = sayingMinInterval;
          }

          lastSayingState = currentState;
        }
      }

      function getSayingState() {
        var injuryLevel = scope.data.injuryLevel || 'Safe';
        var gTotal = scope.data.currentGTotal || 0;
        var speed = scope.data.speed || 0;

        // Check for high G situation first (takes priority)
        if (gTotal > 10) return 'highg';

        // Map injury level to saying state (injuries take priority over airborne)
        if (injuryLevel === 'Fatal') return 'fatal';
        if (injuryLevel === 'Critical') return 'critical';
        if (injuryLevel === 'Severe Injury') return 'severe';
        if (injuryLevel === 'Moderate Injury') return 'moderate';
        if (injuryLevel === 'Minor Injury') return 'minor';

        // Airborne state - flying through the air!
        if (scope.isAirborne && scope.currentAirtime > 0.5) return 'airborne';

        // Idle if not moving
        if (speed < 5 && gTotal < 1) return 'idle';

        return 'safe';
      }

      // Track data for calculations
      var lastUpdateTime = Date.now();
      var highGDuration = 0;
      var sustainedGThreshold = 15;
      var impactCount = 0;

      // Impact detection system
      var lastGTotal = 0;
      var isInImpact = false;
      var impactStartG = 0;
      var timeSinceLastImpact = 0;

      // Track worst state (persists until reset)
      var worstSurvival = 100;
      var worstInjury = 'Safe';

      console.log('crashSurvivalRating: Initialized - worstSurvival:', worstSurvival, 'worstInjury:', worstInjury);

      // G-force thresholds based on NATO RTO-EN-HFM-113 and NHTSA research
      var gforceThresholds = {
        // Lateral (transverse/side) impacts - MOST DANGEROUS
        lateral_safe: 12,
        lateral_serious: 20,
        lateral_severe: 25,
        lateral_fatal: 40,

        // Frontal/Rear (longitudinal) impacts - MODERATE DANGER
        longitudinal_safe: 25,
        longitudinal_serious: 45,
        longitudinal_severe: 60,
        longitudinal_fatal: 75,

        // Vertical impacts - MODERATE DANGER
        vertical_safe: 15,
        vertical_serious: 20,
        vertical_severe: 25,
        vertical_fatal: 40
      };

      // Format G-force values
      scope.formatG = function(value) {
        if (value === undefined || value === null) return '0.00';
        return Math.abs(value).toFixed(2);
      };

      // Format duration
      scope.formatDuration = function(value) {
        if (!value) return '0.00';
        return value.toFixed(2);
      };

      // Get survival rating color class
      scope.getSurvivalClass = function() {
        var rating = scope.data.survivalRating || 100;
        if (rating >= 90) return 'safe';
        if (rating >= 70) return 'minor';
        if (rating >= 50) return 'moderate';
        if (rating >= 30) return 'severe';
        if (rating > 0) return 'critical';
        return 'fatal';
      };

      // Get injury level color class
      scope.getInjuryClass = function() {
        var level = scope.data.injuryLevel || 'Safe';
        if (level === 'Safe') return 'safe';
        if (level === 'Minor Injury') return 'minor';
        if (level === 'Moderate Injury') return 'moderate';
        if (level === 'Severe Injury') return 'severe';
        if (level === 'Critical') return 'critical';
        if (level === 'Fatal') return 'fatal';
        return 'safe';
      };

      // Get class for body part based on injury level
      scope.getBodyPartClass = function(partName) {
        var injury = scope.data.bodyParts[partName] || 0;
        if (injury < 10) return 'safe';
        if (injury < 30) return 'minor';
        if (injury < 50) return 'moderate';
        if (injury < 70) return 'severe';
        if (injury < 90) return 'critical';
        return 'fatal';
      };

      // Get opacity for body part indicator (0.2 to 1.0)
      scope.getBodyPartOpacity = function(partName) {
        var injury = scope.data.bodyParts[partName] || 0;
        return 0.2 + (injury / 100) * 0.8;
      };

      // Get segment class for the 10-segment survival bar
      // Segments fill from left (0) to right (9), drain from right to left
      scope.getSegmentClass = function(index) {
        var rating = scope.data.survivalRating;
        if (rating === undefined || rating === null) rating = 100;

        var injuryLevel = scope.data.injuryLevel || 'Safe';

        // Calculate filled segments (rating 100 = 10 segments, 50 = 5, 0 = 0)
        var filledSegments = Math.ceil(rating / 10);

        // Force 0 segments if Fatal
        if (injuryLevel === 'Fatal' || rating <= 0) {
          filledSegments = 0;
        }

        // Segment is not filled (empty/dark)
        if (index >= filledSegments) {
          return '';
        }

        // Segment is filled - determine color based on rating AND injury level
        var classes = 'filled';

        // Use injury level for color when available (more reliable)
        if (injuryLevel === 'Critical' || rating <= 30) {
          classes += ' critical';
        } else if (injuryLevel === 'Severe Injury' || rating <= 50) {
          classes += ' danger';
        } else if (injuryLevel === 'Moderate Injury' || injuryLevel === 'Minor Injury' || rating <= 70) {
          classes += ' warning';
        }
        // Above 70% with Safe status stays default green

        return classes;
      };

      // Get body part damage class (darkening effect) - LEGACY
      scope.getBodyPartDamageClass = function(partName) {
        var injury = scope.data.bodyParts[partName] || 0;
        if (injury < 15) return '';
        if (injury < 35) return 'injured-light';
        if (injury < 55) return 'injured-moderate';
        if (injury < 75) return 'injured-severe';
        if (injury < 90) return 'injured-critical';
        return 'injured-fatal';
      };

      // Get body part COLOR class (dynamic color palette like health bar)
      // healthy = green, injured = yellow -> orange -> red
      scope.getBodyPartColorClass = function(partName) {
        var injury = scope.data.bodyParts[partName] || 0;
        if (injury < 10) return 'status-healthy';   // Green
        if (injury < 30) return 'status-minor';     // Yellow
        if (injury < 50) return 'status-moderate';  // Orange
        if (injury < 70) return 'status-severe';    // Orange-Red
        if (injury < 90) return 'status-critical';  // Red (pulsing)
        return 'status-fatal';                       // Dark Red (pulsing)
      };

      // Get G-force value color class
      scope.getGForceClass = function(value) {
        var absValue = Math.abs(value || 0);
        if (absValue < 2) return 'gforce-safe';
        if (absValue < 5) return 'gforce-elevated';
        if (absValue < 10) return 'gforce-high';
        return 'gforce-danger';
      };

      // Get G-force segment class for segmented bars
      // Scale: 0-50G (each segment = 5G)
      scope.getGForceSegmentClass = function(index, gValue) {
        var absValue = Math.abs(gValue || 0);
        // Each segment represents 5G (0-5, 5-10, 10-15, etc up to 45-50)
        var filledSegments = Math.min(10, Math.floor(absValue / 5));

        if (index >= filledSegments) {
          return ''; // Empty segment
        }

        var classes = 'filled';

        // Color based on G-force level
        if (absValue >= 20) {
          classes += ' critical';
        } else if (absValue >= 10) {
          classes += ' danger';
        } else if (absValue >= 5) {
          classes += ' warning';
        }
        // Under 5G stays green

        return classes;
      };

      // Get speed segment class for segmented bar
      // Scale: 0-200 (each segment = 20 mph/kph)
      scope.getSpeedSegmentClass = function(index) {
        var speed = scope.data.speed || 0;
        // Each segment represents 20 units (0-20, 20-40, etc up to 180-200)
        var filledSegments = Math.min(10, Math.floor(speed / 20));

        if (index >= filledSegments) {
          return ''; // Empty segment
        }

        var classes = 'filled';

        // Color based on speed
        if (speed >= 150) {
          classes += ' critical';
        } else if (speed >= 100) {
          classes += ' danger';
        } else if (speed >= 60) {
          classes += ' warning';
        }
        // Under 60 stays green

        return classes;
      };

      // Get RPM segment class for segmented bar
      // Scale: 0-10000 (each segment = 1000 RPM)
      scope.getRpmSegmentClass = function(index) {
        var rpm = scope.data.rpm || 0;
        // Each segment represents 1000 RPM
        var filledSegments = Math.min(10, Math.floor(rpm / 1000));

        if (index >= filledSegments) {
          return ''; // Empty segment
        }

        var classes = 'filled';

        // Color based on RPM (redline warning)
        if (rpm >= 7000) {
          classes += ' critical';
        } else if (rpm >= 5500) {
          classes += ' danger';
        } else if (rpm >= 4000) {
          classes += ' warning';
        }
        // Under 4000 stays green

        return classes;
      };

      // Get status display class based on injury level
      scope.getStatusClass = function() {
        var level = scope.data.injuryLevel || 'Safe';
        if (level === 'Safe') return 'status-safe';
        if (level === 'Minor Injury') return 'status-minor';
        if (level === 'Moderate Injury') return 'status-moderate';
        if (level === 'Severe Injury') return 'status-severe';
        if (level === 'Critical') return 'status-critical';
        if (level === 'Fatal') return 'status-fatal';
        return 'status-safe';
      };

      // Get border class for outer container (reflects survival status)
      scope.getBorderClass = function() {
        var level = scope.data.injuryLevel || 'Safe';
        if (level === 'Fatal') return 'border-fatal';
        if (level === 'Critical') return 'border-critical';
        if (level === 'Severe Injury') return 'border-severe';
        if (level === 'Moderate Injury') return 'border-moderate';
        if (level === 'Minor Injury') return 'border-minor';
        return 'border-safe';
      };

      // Settings functions
      scope.toggleSettings = function() {
        scope.showSettings = !scope.showSettings;
      };

      scope.updateSetting = function(key, value) {
        scope.settings[key] = value;
        console.log('crashSurvivalRating: Setting updated -', key, '=', value);
      };

      scope.resetSettings = function() {
        scope.settings = {
          useMetric: false,
          showVehicleStats: true,
          showGForceBreakdown: true,
          showBodyPartIndicators: true,
          soundEnabled: false,
          debugMode: false
        };
        console.log('crashSurvivalRating: Settings reset to defaults');
      };

      scope.saveSettings = function() {
        scope.showSettings = false;
        console.log('crashSurvivalRating: Settings saved');
      };

      // Calculate body part injuries based on G-forces
      function calculateBodyPartInjuries(gx, gy, gz, gTotal) {
        var injuries = scope.data.bodyParts;

        // Head injury - affected by all axes, especially frontal and lateral
        var headBase = Math.abs(gx) * 0.4 + Math.abs(gy) * 0.5 + Math.abs(gz) * 0.3;
        injuries.head = Math.min(100, Math.max(0, (headBase / 30) * 100));

        // Neck injury - whiplash from frontal/rear, lateral stress from side
        var neckBase = Math.abs(gx) * 0.5 + Math.abs(gy) * 0.4 + Math.abs(gz) * 0.2;
        injuries.neck = Math.min(100, Math.max(0, (neckBase / 25) * 100));

        // Chest injury - primarily frontal impacts
        var chestBase = Math.abs(gx) * 0.6 + Math.abs(gy) * 0.2 + Math.abs(gz) * 0.3;
        injuries.chest = Math.min(100, Math.max(0, (chestBase / 35) * 100));

        // Spine injury - vertical compression and lateral bending
        var spineBase = Math.abs(gz) * 0.6 + Math.abs(gy) * 0.3 + Math.abs(gx) * 0.2;
        injuries.spine = Math.min(100, Math.max(0, (spineBase / 25) * 100));

        // Arms - lateral impacts push arms into doors/steering
        injuries.leftArm = Math.min(100, Math.max(0, (Math.abs(gy) * 0.7 + gTotal * 0.2) / 20 * 100));
        injuries.rightArm = Math.min(100, Math.max(0, (Math.abs(gy) * 0.7 + gTotal * 0.2) / 20 * 100));

        // Legs - frontal impacts, pedal intrusion
        var legBase = Math.abs(gx) * 0.5 + gTotal * 0.3;
        injuries.leftLeg = Math.min(100, Math.max(0, (legBase / 30) * 100));
        injuries.rightLeg = Math.min(100, Math.max(0, (legBase / 30) * 100));

        return injuries;
      }

      // Calculate estimated injury probability based on biomechanical research
      function calculateSurvivalRating(gx, gy, gz, gTotal, duration) {
        var survivability = 100;
        var injury = 'Safe';

        var lateralG = Math.abs(gy);
        var verticalG = Math.abs(gz);
        var longitudinalG = Math.abs(gx);

        // Debug: Log when high G detected
        if (gTotal > 10 && scope.settings.debugMode) {
          console.log('CALC INPUT: Total:', gTotal.toFixed(2) + 'G',
                      '| Lateral:', lateralG.toFixed(2) + 'G',
                      '| Long:', longitudinalG.toFixed(2) + 'G',
                      '| Vert:', verticalG.toFixed(2) + 'G');
        }

        // === BASELINE PHYSIOLOGICAL STRESS ===
        if (gTotal > 1.0) {
          var baselineStress = Math.pow((gTotal - 1.0) / 2.0, 1.5) * 0.5;
          survivability -= baselineStress;
        }

        // === LATERAL (SIDE) IMPACT ANALYSIS - MOST CRITICAL ===
        if (lateralG > 3 && lateralG <= gforceThresholds.lateral_safe) {
          var lowLateralPenalty = Math.pow((lateralG - 3) / 9, 2) * 5;
          survivability -= lowLateralPenalty;
        }

        if (lateralG > gforceThresholds.lateral_safe) {
          if (lateralG >= gforceThresholds.lateral_fatal) {
            survivability -= 85;
            injury = 'Fatal';
          } else if (lateralG >= gforceThresholds.lateral_severe) {
            var severity = (lateralG - gforceThresholds.lateral_severe) /
                          (gforceThresholds.lateral_fatal - gforceThresholds.lateral_severe);
            survivability -= 50 + (severity * 30);
            injury = 'Critical';
          } else if (lateralG >= gforceThresholds.lateral_serious) {
            var severity = (lateralG - gforceThresholds.lateral_serious) /
                          (gforceThresholds.lateral_severe - gforceThresholds.lateral_serious);
            survivability -= 30 + (severity * 20);
            if (injury === 'Safe') injury = 'Severe Injury';
          } else {
            var severity = (lateralG - gforceThresholds.lateral_safe) /
                          (gforceThresholds.lateral_serious - gforceThresholds.lateral_safe);
            survivability -= 15 + (severity * 15);
            if (injury === 'Safe') injury = 'Moderate Injury';
          }
        }

        // === LONGITUDINAL (FRONT/REAR) IMPACT ANALYSIS ===
        if (longitudinalG > 5 && longitudinalG <= gforceThresholds.longitudinal_safe) {
          var lowLongitudinalPenalty = Math.pow((longitudinalG - 5) / 20, 2) * 3;
          survivability -= lowLongitudinalPenalty;
        }

        if (longitudinalG > gforceThresholds.longitudinal_safe) {
          if (longitudinalG >= gforceThresholds.longitudinal_fatal) {
            survivability -= 70;
            injury = 'Fatal';
          } else if (longitudinalG >= gforceThresholds.longitudinal_severe) {
            var severity = (longitudinalG - gforceThresholds.longitudinal_severe) /
                          (gforceThresholds.longitudinal_fatal - gforceThresholds.longitudinal_severe);
            survivability -= 40 + (severity * 25);
            if (injury !== 'Fatal' && injury !== 'Critical') injury = 'Critical';
          } else if (longitudinalG >= gforceThresholds.longitudinal_serious) {
            var severity = (longitudinalG - gforceThresholds.longitudinal_serious) /
                          (gforceThresholds.longitudinal_severe - gforceThresholds.longitudinal_serious);
            survivability -= 20 + (severity * 20);
            if (injury === 'Safe' || injury === 'Moderate Injury') injury = 'Severe Injury';
          } else {
            var severity = (longitudinalG - gforceThresholds.longitudinal_safe) /
                          (gforceThresholds.longitudinal_serious - gforceThresholds.longitudinal_safe);
            survivability -= 10 + (severity * 10);
            if (injury === 'Safe') injury = 'Minor Injury';
          }
        }

        // === VERTICAL IMPACT ANALYSIS ===
        if (verticalG > 3 && verticalG <= gforceThresholds.vertical_safe) {
          var lowVerticalPenalty = Math.pow((verticalG - 3) / 12, 2) * 4;
          survivability -= lowVerticalPenalty;
        }

        if (verticalG > gforceThresholds.vertical_safe) {
          if (verticalG >= gforceThresholds.vertical_fatal) {
            survivability -= 60;
            if (injury !== 'Fatal') injury = 'Critical';
          } else if (verticalG >= gforceThresholds.vertical_severe) {
            var severity = (verticalG - gforceThresholds.vertical_severe) /
                          (gforceThresholds.vertical_fatal - gforceThresholds.vertical_severe);
            survivability -= 30 + (severity * 25);
            if (injury === 'Safe' || injury === 'Minor Injury') injury = 'Severe Injury';
          } else if (verticalG >= gforceThresholds.vertical_serious) {
            var severity = (verticalG - gforceThresholds.vertical_serious) /
                          (gforceThresholds.vertical_severe - gforceThresholds.vertical_serious);
            survivability -= 15 + (severity * 15);
            if (injury === 'Safe') injury = 'Moderate Injury';
          } else {
            var severity = (verticalG - gforceThresholds.vertical_safe) /
                          (gforceThresholds.vertical_serious - gforceThresholds.vertical_safe);
            survivability -= 8 + (severity * 7);
            if (injury === 'Safe') injury = 'Minor Injury';
          }
        }

        // === DURATION EFFECTS (Wayne State Tolerance Curve) ===
        if (duration > 0.015) {
          if (duration > 0.2) {
            var durationFactor = Math.min(duration / 0.2, 3);
            survivability -= (100 - survivability) * 0.3 * durationFactor;
          } else if (duration > 0.05) {
            var durationFactor = (duration - 0.05) / 0.15;
            survivability -= (100 - survivability) * 0.15 * durationFactor;
          }
        }

        // === MULTIPLE IMPACT PENALTY ===
        if (impactCount > 1) {
          var multiImpactPenalty = Math.min((impactCount - 1) * 8, 30);
          survivability -= multiImpactPenalty;
        }

        // Ensure survivability stays within bounds
        survivability = Math.max(0, Math.min(100, survivability));

        // === FINAL TWO-WAY SYNCHRONIZATION ===
        if (injury === 'Fatal') {
          survivability = 0;
        } else if (injury === 'Critical' && survivability > 29) {
          survivability = Math.min(survivability, 29);
        } else if (injury === 'Severe Injury' && survivability > 49) {
          survivability = Math.min(survivability, 49);
        } else if (injury === 'Moderate Injury' && survivability > 69) {
          survivability = Math.min(survivability, 69);
        } else if (injury === 'Minor Injury' && survivability > 89) {
          survivability = Math.min(survivability, 89);
        }

        // If survival is low, upgrade injury level to match
        if (survivability <= 0) {
          injury = 'Fatal';
        } else if (survivability < 30) {
          if (injury !== 'Fatal') injury = 'Critical';
        } else if (survivability < 50) {
          if (injury !== 'Fatal' && injury !== 'Critical') injury = 'Severe Injury';
        } else if (survivability < 70) {
          if (injury !== 'Fatal' && injury !== 'Critical' && injury !== 'Severe Injury') injury = 'Moderate Injury';
        } else if (survivability < 90) {
          if (injury === 'Safe') injury = 'Minor Injury';
        }

        return {
          rating: survivability,
          injury: injury
        };
      }

      // Listen for streamsUpdate to get live acceleration data
      scope.$on('streamsUpdate', function (event, streams) {
        if (!streams.electrics) return;

        var electrics = streams.electrics;

        // Debug logging every 60 updates (~2 seconds)
        updateCount++;
        if (updateCount % 60 === 0 && scope.settings.debugMode) {
          console.log('crashSurvivalRating: Update', updateCount,
                      'GTotal:', scope.data.currentGTotal?.toFixed(2),
                      'MaxG:', scope.data.maxGTotal?.toFixed(2),
                      'Survival:', scope.data.survivalRating);
        }

        // Get acceleration values (m/s²)
        var ax = electrics.accXSmooth || 0;
        var ay = electrics.accYSmooth || 0;
        var az = electrics.accZSmooth || 0;

        // Calculate magnitude FIRST
        var accelMag = Math.sqrt(ax*ax + ay*ay + az*az);

        // Convert to G-forces (1G = 9.81 m/s²)
        var currentGx = ax / 9.81;
        var currentGy = ay / 9.81;
        var currentGz = az / 9.81;
        var currentGTotal = accelMag / 9.81;

        // Update current values
        scope.data.currentGx = currentGx;
        scope.data.currentGy = currentGy;
        scope.data.currentGz = currentGz;
        scope.data.currentGTotal = currentGTotal;

        // Update vehicle stats from electrics stream
        var speedMultiplier = scope.settings.useMetric ? 3.6 : 2.23694;
        scope.data.speed = Math.round((electrics.wheelspeed || electrics.airspeed || 0) * speedMultiplier);
        scope.data.speedUnit = scope.settings.useMetric ? 'KPH' : 'MPH';
        scope.data.rpm = Math.round(electrics.rpm || 0);
        scope.data.gear = electrics.gear || 'N';
        scope.data.fuel = Math.round((electrics.fuel || 0) * 100);
        scope.data.throttle = Math.round((electrics.throttle || 0) * 100);
        scope.data.brake = Math.round((electrics.brake || 0) * 100);
        scope.data.engineTemp = Math.round(electrics.watertemp || electrics.oiltemp || 0);

        // === AIRTIME DETECTION ===
        // Detect airborne: wheels not spinning but vehicle moving, or very low vertical G (freefall)
        var wheelspeed = electrics.wheelspeed || 0;
        var airspeed = electrics.airspeed || 0;
        var verticalG = Math.abs(currentGz);

        // Airborne conditions:
        // 1. Vertical G is very low (freefall = ~0G) while moving
        // 2. Wheel speed is much lower than air speed (wheels not touching ground)
        var isCurrentlyAirborne = false;

        if (airspeed > 5) { // Must be moving
          // Check for freefall (vertical G < 0.3G is basically floating)
          if (verticalG < 0.3) {
            isCurrentlyAirborne = true;
          }
          // Or wheels not spinning while moving fast
          else if (wheelspeed < 1 && airspeed > 10) {
            isCurrentlyAirborne = true;
          }
        }

        scope.isAirborne = isCurrentlyAirborne;

        // Track airtime
        var now = Date.now();
        if (isCurrentlyAirborne) {
          if (!wasAirborne) {
            // Just became airborne
            airtimeStart = now;
          }
          scope.currentAirtime = (now - airtimeStart) / 1000;

          // Update max airtime
          if (scope.currentAirtime > scope.maxAirtime) {
            scope.maxAirtime = scope.currentAirtime;
          }
        } else {
          if (wasAirborne && scope.currentAirtime > 0.3) {
            // Just landed after significant air - trigger crash animation!
            console.log('LANDED! Airtime:', scope.currentAirtime.toFixed(2) + 's');

            // Trigger rocket crash animation
            scope.rocketCrashed = true;

            // Clear any existing timeout
            if (crashTimeout) {
              clearTimeout(crashTimeout);
            }

            // Reset crash state after animation completes (0.8 seconds)
            crashTimeout = setTimeout(function() {
              scope.rocketCrashed = false;
              scope.$apply();
            }, 800);
          }
          // Reset current airtime when grounded
          scope.currentAirtime = 0;
        }
        wasAirborne = isCurrentlyAirborne;

        // Track maximum G-forces
        if (Math.abs(currentGx) > Math.abs(scope.data.maxGx)) {
          scope.data.maxGx = currentGx;
        }
        if (Math.abs(currentGy) > Math.abs(scope.data.maxGy)) {
          scope.data.maxGy = currentGy;
        }
        if (Math.abs(currentGz) > Math.abs(scope.data.maxGz)) {
          scope.data.maxGz = currentGz;
        }
        if (currentGTotal > scope.data.maxGTotal) {
          scope.data.maxGTotal = currentGTotal;
        }

        // === DISCRETE IMPACT DETECTION (HIT-BY-HIT) ===
        var impactThreshold = 5;
        var impactCooldown = 0.2;
        var gForceRise = currentGTotal - lastGTotal;

        // Track time since last impact
        var now = Date.now();
        var dt = (now - lastUpdateTime) / 1000;
        timeSinceLastImpact += dt;

        // Detect NEW impact
        if (!isInImpact && gForceRise > 3 && currentGTotal > impactThreshold && timeSinceLastImpact > impactCooldown) {
          isInImpact = true;
          impactStartG = currentGTotal;
          impactCount++;
          scope.data.impactCount = impactCount;
          timeSinceLastImpact = 0;

          console.log('IMPACT DETECTED:', impactCount, 'Severity:', currentGTotal.toFixed(2) + 'G');
        }

        // End impact when G-force drops significantly
        if (isInImpact && currentGTotal < impactStartG * 0.5) {
          isInImpact = false;
        }

        lastGTotal = currentGTotal;

        // Track duration of high G-forces
        lastUpdateTime = now;

        if (currentGTotal > sustainedGThreshold) {
          highGDuration += dt;
        } else {
          highGDuration = 0;
        }
        scope.data.highGDuration = highGDuration;

        // Calculate survivability
        var result = calculateSurvivalRating(
          scope.data.maxGx,
          scope.data.maxGy,
          scope.data.maxGz,
          scope.data.maxGTotal,
          highGDuration
        );

        // Calculate body part injuries
        calculateBodyPartInjuries(
          scope.data.maxGx,
          scope.data.maxGy,
          scope.data.maxGz,
          scope.data.maxGTotal
        );

        // === TRACK WORST STATE ===
        var injuryRanking = {'Safe': 0, 'Minor Injury': 1, 'Moderate Injury': 2, 'Severe Injury': 3, 'Critical': 4, 'Fatal': 5};

        // Update worst survival (lower is worse)
        if (result.rating < worstSurvival) {
          worstSurvival = result.rating;
        }

        // Update worst injury (higher rank is worse)
        if (injuryRanking[result.injury] > injuryRanking[worstInjury]) {
          worstInjury = result.injury;
        }

        // Force sync with explicit if statements
        if (worstInjury === 'Fatal') {
          worstSurvival = 0;
        } else if (worstInjury === 'Critical') {
          worstSurvival = Math.min(worstSurvival, 29);
        } else if (worstInjury === 'Severe Injury') {
          worstSurvival = Math.min(worstSurvival, 49);
        } else if (worstInjury === 'Moderate Injury') {
          worstSurvival = Math.min(worstSurvival, 69);
        } else if (worstInjury === 'Minor Injury') {
          worstSurvival = Math.min(worstSurvival, 89);
        }

        // Apply to display
        scope.data.survivalRating = Math.floor(worstSurvival);
        scope.data.injuryLevel = worstInjury;

        // Absolute failsafe
        if (scope.data.injuryLevel === 'Fatal' && scope.data.survivalRating !== 0) {
          scope.data.survivalRating = 0;
        }

        // Update character saying based on current state
        sayingCooldown -= dt;
        var currentSayingState = getSayingState();
        updateSaying(currentSayingState);
      });

      // Handle vehicle reset
      scope.$on('VehicleReset', function() {
        console.log('crashSurvivalRating: VehicleReset event received!');
        scope.data = {
          currentGx: 0,
          currentGy: 0,
          currentGz: 0,
          currentGTotal: 0,
          maxGx: 0,
          maxGy: 0,
          maxGz: 0,
          maxGTotal: 0,
          survivalRating: 100,
          injuryLevel: 'Safe',
          highGDuration: 0,
          impactCount: 0,
          bodyParts: {
            head: 0,
            neck: 0,
            chest: 0,
            spine: 0,
            leftArm: 0,
            rightArm: 0,
            leftLeg: 0,
            rightLeg: 0
          },
          speed: 0,
          speedUnit: scope.settings.useMetric ? 'KPH' : 'MPH',
          rpm: 0,
          gear: 'N',
          fuel: 0,
          throttle: 0,
          brake: 0,
          engineTemp: 0
        };
        highGDuration = 0;
        impactCount = 0;
        lastGTotal = 0;
        isInImpact = false;
        impactStartG = 0;
        timeSinceLastImpact = 0;
        worstSurvival = 100;
        worstInjury = 'Safe';
        lastUpdateTime = Date.now();
        // Reset saying and typewriter
        lastSayingState = '';
        sayingCooldown = 0;
        if (typewriterInterval) {
          clearInterval(typewriterInterval);
          typewriterInterval = null;
        }
        startTypewriter("Ready to roll!");

        // Reset heart rate
        scope.heartRate = 72;
        targetHeartRate = 72;
        ecgDotX = 0;
        ecgPhase = 0;
        ecgTrail = [];

        // Reset airtime
        scope.isAirborne = false;
        scope.currentAirtime = 0;
        scope.maxAirtime = 0;
        scope.rocketCrashed = false;
        airtimeStart = 0;
        wasAirborne = false;
        if (crashTimeout) {
          clearTimeout(crashTimeout);
          crashTimeout = null;
        }

        console.log('crashSurvivalRating: Reset complete - worstSurvival:', worstSurvival, 'worstInjury:', worstInjury);
      });

      // Cleanup
      scope.$on('$destroy', function() {
        // Clean up resize observer
        if (resizeObserver) {
          resizeObserver.disconnect();
        }
        if (typeof resizeInterval !== 'undefined') {
          clearInterval(resizeInterval);
        }

        // Clean up typewriter interval
        if (typewriterInterval) {
          clearInterval(typewriterInterval);
          typewriterInterval = null;
        }

        // Clean up ECG animation
        if (ecgAnimationFrame) {
          cancelAnimationFrame(ecgAnimationFrame);
          ecgAnimationFrame = null;
        }

        // Clean up crash timeout
        if (crashTimeout) {
          clearTimeout(crashTimeout);
          crashTimeout = null;
        }

        // Clean up streams
        if (typeof StreamsManager !== 'undefined') {
          StreamsManager.remove(streamsList);
        } else if (scope.StreamsManager) {
          scope.StreamsManager.remove(streamsList);
        }
      });

      // Initial typewriter message
      startTypewriter("Ready to roll!");

      console.log('crashSurvivalRating: UI App ready');
    }
  };
}]);
