-- ============================================================================
-- Crash Survival Rating Pro v3.0 - Vehicle Extension
-- Copyright (c) 2025 minMAX420 - All Rights Reserved
--
-- Licensed under CC BY-NC-SA 4.0 (Creative Commons Attribution-NonCommercial-ShareAlike)
-- https://creativecommons.org/licenses/by-nc-sa/4.0/
--
-- PERSONAL USE ONLY - Commercial use is strictly prohibited.
-- See LICENSE file for full terms.
-- ============================================================================
--
-- Advanced G-Force and Crash Survivability Monitor
-- Based on real-world crash test data and human tolerance research
-- Sources: NATO RTO-EN-HFM-113, NHTSA FMVSS 208
--
-- ARCHITECTURE:
-- This Lua extension is the authoritative source for all survivability calculations.
-- The UI (app.js) displays the data sent via guihooks.trigger().
-- Per BeamNG best practices: "JavaScript only for UI display purposes,
-- leaving as much logic as possible to be calculated in Lua."

local M = {}

-- ============================================================================
-- SETTINGS (loaded from GE extension settings module)
-- ============================================================================
local settings = {
    updateFrequency = 20,
    impactThreshold = 5.0,
    impactCooldown = 0.2,
    highGThreshold = 4.0,
    highGDurationWarn = 0.5,
    smoothingFactor = 0.3,
    useMetric = false,
    debugMode = false,
    thresholds = {
        minor = 4.0,
        moderate = 8.0,
        severe = 15.0,
        critical = 25.0,
        fatal = 40.0
    }
}

-- ============================================================================
-- UPDATE TIMERS
-- ============================================================================
local updateTimer = 0
local updateInterval = 0.05  -- Will be recalculated from settings
local debugTimer = 0
local debugInterval = 2.0

-- ============================================================================
-- G-FORCE TRACKING
-- ============================================================================
local currentGx = 0
local currentGy = 0
local currentGz = 0
local currentGTotal = 0

local maxGx = 0
local maxGy = 0
local maxGz = 0
local maxGTotal = 0

-- Smoothed values for display
local smoothGx = 0
local smoothGy = 0
local smoothGz = 0
local smoothGTotal = 0

-- Duration tracking
local highGDuration = 0
local sustainedGThreshold = 15

-- ============================================================================
-- IMPACT DETECTION
-- ============================================================================
local lastGTotal = 0
local isInImpact = false
local impactStartG = 0
local timeSinceLastImpact = 0
local impactCount = 0

-- ============================================================================
-- SURVIVABILITY DATA
-- ============================================================================
local survivalRating = 100
local injuryLevel = "Safe"
local worstSurvival = 100
local worstInjury = "Safe"

-- ============================================================================
-- BODY PART INJURY TRACKING
-- ============================================================================
local bodyPartInjuries = {
    head = 0,       -- 0-100, 0 = no injury
    neck = 0,
    chest = 0,
    spine = 0,
    leftArm = 0,
    rightArm = 0,
    leftLeg = 0,
    rightLeg = 0
}

-- ============================================================================
-- REUSABLE DATA TABLE (reduces GC pressure)
-- ============================================================================
local dataPacket = {
    currentGx = 0,
    currentGy = 0,
    currentGz = 0,
    currentGTotal = 0,
    maxGx = 0,
    maxGy = 0,
    maxGz = 0,
    maxGTotal = 0,
    survivalRating = 100,
    injuryLevel = "Safe",
    worstSurvival = 100,
    worstInjury = "Safe",
    highGDuration = 0,
    impactCount = 0,
    bodyParts = {},
    speed = 0,
    speedUnit = "MPH",
    rpm = 0,
    gear = "N",
    fuel = 100,
    throttle = 0,
    brake = 0,
    engineTemp = 0
}

-- ============================================================================
-- G-FORCE THRESHOLDS (NATO RTO-EN-HFM-113 and NHTSA research)
-- ============================================================================
local gforceThresholds = {
    -- Lateral (side) impacts - MOST DANGEROUS
    lateral_safe = 12,
    lateral_serious = 20,
    lateral_severe = 25,
    lateral_fatal = 40,

    -- Frontal/Rear (longitudinal) impacts
    longitudinal_safe = 25,
    longitudinal_serious = 45,
    longitudinal_severe = 60,
    longitudinal_fatal = 75,

    -- Vertical impacts
    vertical_safe = 15,
    vertical_serious = 20,
    vertical_severe = 25,
    vertical_fatal = 40,

    -- Overall thresholds
    safe = 15,
    minor_injury = 25,
    moderate_injury = 40,
    severe_injury = 60,
    critical = 80
}

-- ============================================================================
-- HELPER FUNCTIONS
-- ============================================================================

local function round2(num)
    return math.floor(num * 100 + 0.5) / 100
end

local function clamp(val, min, max)
    return math.max(min, math.min(max, val))
end

local function lerp(a, b, t)
    return a + (b - a) * t
end

-- ============================================================================
-- BODY PART INJURY CALCULATION
-- ============================================================================
local function calculateBodyPartInjuries(gx, gy, gz, gTotal)
    local injuries = bodyPartInjuries

    -- Head injury - affected by all axes, especially frontal and lateral
    local headBase = math.abs(gx) * 0.4 + math.abs(gy) * 0.5 + math.abs(gz) * 0.3
    injuries.head = clamp((headBase / 30) * 100, 0, 100)

    -- Neck injury - whiplash from frontal/rear, lateral stress from side
    local neckBase = math.abs(gx) * 0.5 + math.abs(gy) * 0.4 + math.abs(gz) * 0.2
    injuries.neck = clamp((neckBase / 25) * 100, 0, 100)

    -- Chest injury - primarily frontal impacts
    local chestBase = math.abs(gx) * 0.6 + math.abs(gy) * 0.2 + math.abs(gz) * 0.3
    injuries.chest = clamp((chestBase / 35) * 100, 0, 100)

    -- Spine injury - vertical compression and lateral bending
    local spineBase = math.abs(gz) * 0.6 + math.abs(gy) * 0.3 + math.abs(gx) * 0.2
    injuries.spine = clamp((spineBase / 25) * 100, 0, 100)

    -- Arms - lateral impacts push arms into doors/steering
    injuries.leftArm = clamp((math.abs(gy) * 0.7 + gTotal * 0.2) / 20 * 100, 0, 100)
    injuries.rightArm = clamp((math.abs(gy) * 0.7 + gTotal * 0.2) / 20 * 100, 0, 100)

    -- Legs - frontal impacts, pedal intrusion
    local legBase = math.abs(gx) * 0.5 + gTotal * 0.3
    injuries.leftLeg = clamp((legBase / 30) * 100, 0, 100)
    injuries.rightLeg = clamp((legBase / 30) * 100, 0, 100)

    return injuries
end

-- ============================================================================
-- SURVIVABILITY CALCULATION
-- ============================================================================
local function calculateSurvivalRating(gx, gy, gz, gTotal)
    local rating = 100
    local injury = "Safe"

    local lateralG = math.abs(gy)
    local verticalG = math.abs(gz)
    local longitudinalG = math.abs(gx)

    -- Baseline physiological stress from ANY G-force above 1G
    if gTotal > 1.0 then
        local baselineStress = math.pow((gTotal - 1.0) / 2.0, 1.5) * 0.5
        rating = rating - baselineStress
    end

    -- Low-moderate lateral forces (3-12G)
    if lateralG > 3 and lateralG <= gforceThresholds.lateral_safe then
        local lowLateralPenalty = math.pow((lateralG - 3) / 9, 2) * 5
        rating = rating - lowLateralPenalty
    end

    -- Lateral (side) impacts - most dangerous (NATO: 40G absolute limit)
    if lateralG > gforceThresholds.lateral_safe then
        if lateralG >= gforceThresholds.lateral_fatal then
            rating = rating - 85
            injury = "Fatal"
        elseif lateralG >= gforceThresholds.lateral_severe then
            local severity = (lateralG - gforceThresholds.lateral_severe) /
                            (gforceThresholds.lateral_fatal - gforceThresholds.lateral_severe)
            rating = rating - (50 + severity * 30)
            injury = "Critical"
        elseif lateralG >= gforceThresholds.lateral_serious then
            local severity = (lateralG - gforceThresholds.lateral_serious) /
                            (gforceThresholds.lateral_severe - gforceThresholds.lateral_serious)
            rating = rating - (30 + severity * 20)
            if injury == "Safe" then injury = "Severe" end
        else
            local severity = (lateralG - gforceThresholds.lateral_safe) /
                            (gforceThresholds.lateral_serious - gforceThresholds.lateral_safe)
            rating = rating - (15 + severity * 15)
            if injury == "Safe" then injury = "Moderate" end
        end
    end

    -- Low-moderate longitudinal forces (5-25G)
    if longitudinalG > 5 and longitudinalG <= gforceThresholds.longitudinal_safe then
        local lowLongitudinalPenalty = math.pow((longitudinalG - 5) / 20, 2) * 3
        rating = rating - lowLongitudinalPenalty
    end

    -- Longitudinal (forward/back) impacts (NHTSA: 75G limit)
    if longitudinalG > gforceThresholds.longitudinal_safe then
        if longitudinalG >= gforceThresholds.longitudinal_fatal then
            rating = rating - 70
            injury = "Fatal"
        elseif longitudinalG >= gforceThresholds.longitudinal_severe then
            local severity = (longitudinalG - gforceThresholds.longitudinal_severe) /
                            (gforceThresholds.longitudinal_fatal - gforceThresholds.longitudinal_severe)
            rating = rating - (40 + severity * 25)
            if injury ~= "Fatal" and injury ~= "Critical" then injury = "Critical" end
        elseif longitudinalG >= gforceThresholds.longitudinal_serious then
            local severity = (longitudinalG - gforceThresholds.longitudinal_serious) /
                            (gforceThresholds.longitudinal_severe - gforceThresholds.longitudinal_serious)
            rating = rating - (20 + severity * 20)
            if injury == "Safe" or injury == "Moderate" then injury = "Severe" end
        else
            local severity = (longitudinalG - gforceThresholds.longitudinal_safe) /
                            (gforceThresholds.longitudinal_serious - gforceThresholds.longitudinal_safe)
            rating = rating - (10 + severity * 10)
            if injury == "Safe" then injury = "Minor" end
        end
    end

    -- Low-moderate vertical forces (3-15G)
    if verticalG > 3 and verticalG <= gforceThresholds.vertical_safe then
        local lowVerticalPenalty = math.pow((verticalG - 3) / 12, 2) * 4
        rating = rating - lowVerticalPenalty
    end

    -- Vertical impacts (spinal compression)
    if verticalG > gforceThresholds.vertical_safe then
        if verticalG >= gforceThresholds.vertical_fatal then
            rating = rating - 60
            if injury ~= "Fatal" then injury = "Critical" end
        elseif verticalG >= gforceThresholds.vertical_severe then
            local severity = (verticalG - gforceThresholds.vertical_severe) /
                            (gforceThresholds.vertical_fatal - gforceThresholds.vertical_severe)
            rating = rating - (30 + severity * 25)
            if injury == "Safe" or injury == "Minor" then injury = "Severe" end
        elseif verticalG >= gforceThresholds.vertical_serious then
            local severity = (verticalG - gforceThresholds.vertical_serious) /
                            (gforceThresholds.vertical_severe - gforceThresholds.vertical_serious)
            rating = rating - (15 + severity * 15)
            if injury == "Safe" then injury = "Moderate" end
        else
            local severity = (verticalG - gforceThresholds.vertical_safe) /
                            (gforceThresholds.vertical_serious - gforceThresholds.vertical_safe)
            rating = rating - (8 + severity * 7)
            if injury == "Safe" then injury = "Minor" end
        end
    end

    -- Overall magnitude penalty
    if gTotal > gforceThresholds.safe then
        if gTotal > gforceThresholds.critical then
            rating = math.max(0, rating - 40)
            injury = "Fatal"
        elseif gTotal > gforceThresholds.severe_injury then
            rating = math.max(5, rating - 30)
            if injury ~= "Fatal" then injury = "Critical" end
        elseif gTotal > gforceThresholds.moderate_injury then
            rating = rating - 20
            if injury == "Safe" or injury == "Minor" then injury = "Moderate" end
        elseif gTotal > gforceThresholds.minor_injury then
            rating = rating - 10
            if injury == "Safe" then injury = "Minor" end
        end
    end

    -- Duration penalty - sustained high G-forces are more dangerous
    if highGDuration > 0.5 then
        local durationMultiplier = 1 + (highGDuration - 0.5) * 0.5
        local currentPenalty = (100 - rating) * (durationMultiplier - 1)
        rating = rating - currentPenalty

        if lateralG > 12 and highGDuration > 1.0 then
            rating = math.max(0, rating - 25)
            injury = "Fatal"
        end
    end

    -- Multiple impacts are cumulative
    if impactCount > 1 then
        local multiImpactPenalty = (impactCount - 1) * 5
        rating = rating - multiImpactPenalty
    end

    return clamp(rating, 0, 100), injury
end

-- ============================================================================
-- G-FORCE UPDATE
-- ============================================================================
local function updateGForces(dt)
    if not electrics or not electrics.values then
        return
    end

    -- Get acceleration from electrics values (m/s²)
    local ax = electrics.values.accXSmooth or 0
    local ay = electrics.values.accYSmooth or 0
    local az = electrics.values.accZSmooth or 0

    -- Calculate magnitude
    local accelMag = math.sqrt(ax*ax + ay*ay + az*az)

    -- Convert to G-forces (1G = 9.81 m/s²)
    currentGx = ax / 9.81
    currentGy = ay / 9.81
    currentGz = az / 9.81
    currentGTotal = accelMag / 9.81

    -- Apply smoothing for display
    local sf = settings.smoothingFactor
    smoothGx = lerp(smoothGx, currentGx, 1 - sf)
    smoothGy = lerp(smoothGy, currentGy, 1 - sf)
    smoothGz = lerp(smoothGz, currentGz, 1 - sf)
    smoothGTotal = lerp(smoothGTotal, currentGTotal, 1 - sf)

    -- Track maximum G-forces
    if math.abs(currentGx) > math.abs(maxGx) then maxGx = currentGx end
    if math.abs(currentGy) > math.abs(maxGy) then maxGy = currentGy end
    if math.abs(currentGz) > math.abs(maxGz) then maxGz = currentGz end
    if currentGTotal > maxGTotal then maxGTotal = currentGTotal end

    -- Track duration of high G-forces
    if currentGTotal > settings.highGThreshold then
        highGDuration = highGDuration + dt
    else
        highGDuration = math.max(0, highGDuration - dt * 0.5)  -- Gradual recovery
    end

    -- === IMPACT DETECTION ===
    local gForceRise = currentGTotal - lastGTotal
    timeSinceLastImpact = timeSinceLastImpact + dt

    if not isInImpact and gForceRise > 3 and currentGTotal > settings.impactThreshold and timeSinceLastImpact > settings.impactCooldown then
        isInImpact = true
        impactStartG = currentGTotal
        impactCount = impactCount + 1
        timeSinceLastImpact = 0

        if settings.debugMode then
            print(string.format("crashSurvivalRating: IMPACT #%d - %.2fG", impactCount, currentGTotal))
        end
    end

    if isInImpact and currentGTotal < impactStartG * 0.5 then
        isInImpact = false
    end

    lastGTotal = currentGTotal

    -- Calculate survivability
    survivalRating, injuryLevel = calculateSurvivalRating(maxGx, maxGy, maxGz, maxGTotal)

    -- Track worst state
    if survivalRating < worstSurvival then
        worstSurvival = survivalRating
        worstInjury = injuryLevel
    end

    -- Calculate body part injuries
    calculateBodyPartInjuries(maxGx, maxGy, maxGz, maxGTotal)

    -- Get vehicle stats
    local speed = electrics.values.wheelspeed or electrics.values.airspeed or 0
    if settings.useMetric then
        speed = speed * 3.6  -- m/s to km/h
    else
        speed = speed * 2.237  -- m/s to mph
    end

    -- Update the reusable data packet
    dataPacket.currentGx = round2(smoothGx)
    dataPacket.currentGy = round2(smoothGy)
    dataPacket.currentGz = round2(smoothGz)
    dataPacket.currentGTotal = round2(smoothGTotal)
    dataPacket.maxGx = round2(maxGx)
    dataPacket.maxGy = round2(maxGy)
    dataPacket.maxGz = round2(maxGz)
    dataPacket.maxGTotal = round2(maxGTotal)
    dataPacket.survivalRating = math.floor(survivalRating)
    dataPacket.injuryLevel = injuryLevel
    dataPacket.worstSurvival = math.floor(worstSurvival)
    dataPacket.worstInjury = worstInjury
    dataPacket.highGDuration = round2(highGDuration)
    dataPacket.impactCount = impactCount

    -- Body part injuries
    dataPacket.bodyParts = {
        head = math.floor(bodyPartInjuries.head),
        neck = math.floor(bodyPartInjuries.neck),
        chest = math.floor(bodyPartInjuries.chest),
        spine = math.floor(bodyPartInjuries.spine),
        leftArm = math.floor(bodyPartInjuries.leftArm),
        rightArm = math.floor(bodyPartInjuries.rightArm),
        leftLeg = math.floor(bodyPartInjuries.leftLeg),
        rightLeg = math.floor(bodyPartInjuries.rightLeg)
    }

    -- Vehicle stats
    dataPacket.speed = math.floor(speed)
    dataPacket.speedUnit = settings.useMetric and "KPH" or "MPH"
    dataPacket.rpm = math.floor(electrics.values.rpm or 0)
    dataPacket.gear = electrics.values.gear or "N"
    dataPacket.fuel = math.floor((electrics.values.fuel or 1) * 100)
    dataPacket.throttle = math.floor((electrics.values.throttle or 0) * 100)
    dataPacket.brake = math.floor((electrics.values.brake or 0) * 100)
    dataPacket.engineTemp = math.floor(electrics.values.watertemp or electrics.values.oiltemp or 0)

    -- Vehicle damage integration (optional beamstate data)
    if beamstate then
        -- Get overall vehicle damage percentage
        local damageData = beamstate.getDamage and beamstate.getDamage() or nil
        if damageData then
            dataPacket.vehicleDamage = math.floor((damageData.damage or 0) * 100)
            dataPacket.deformationEnergy = round2(damageData.deformGroupDamage or 0)
        else
            dataPacket.vehicleDamage = 0
            dataPacket.deformationEnergy = 0
        end

        -- Check for low pressure (indicator of serious crash)
        local lowPressure = beamstate.lowpressure or 0
        dataPacket.lowPressure = lowPressure > 0
    else
        dataPacket.vehicleDamage = 0
        dataPacket.deformationEnergy = 0
        dataPacket.lowPressure = false
    end

    -- Safety system states from electrics (if available)
    dataPacket.hazardLights = electrics.values.hazard_enabled or false
    dataPacket.absActive = electrics.values.abs or 0 > 0
    dataPacket.escActive = electrics.values.esc or 0 > 0

    -- Send to UI
    if guihooks then
        guihooks.trigger('CrashSurvivalRatingUpdate', dataPacket)
    end

    -- Debug output
    if settings.debugMode then
        debugTimer = debugTimer + dt
        if debugTimer >= debugInterval then
            if currentGTotal > 1 then
                print(string.format("crashSurvivalRating: G=%.2f Max=%.2f Survival=%d%% Impacts=%d",
                    currentGTotal, maxGTotal, survivalRating, impactCount))
            end
            debugTimer = 0
        end
    end
end

-- ============================================================================
-- MAIN UPDATE
-- ============================================================================
local function onUpdate(dt)
    updateTimer = updateTimer + dt
    if updateTimer >= updateInterval then
        updateGForces(updateTimer)
        updateTimer = 0
    end
end

-- ============================================================================
-- RESET
-- ============================================================================
local function onReset()
    maxGx = 0
    maxGy = 0
    maxGz = 0
    maxGTotal = 0
    currentGx = 0
    currentGy = 0
    currentGz = 0
    currentGTotal = 0
    smoothGx = 0
    smoothGy = 0
    smoothGz = 0
    smoothGTotal = 0
    survivalRating = 100
    worstSurvival = 100
    injuryLevel = "Safe"
    worstInjury = "Safe"
    highGDuration = 0
    updateTimer = 0
    debugTimer = 0
    impactCount = 0
    lastGTotal = 0
    isInImpact = false
    impactStartG = 0
    timeSinceLastImpact = 0

    -- Reset body parts
    for k in pairs(bodyPartInjuries) do
        bodyPartInjuries[k] = 0
    end

    print("crashSurvivalRating: State reset")

    -- Notify UI
    if guihooks then
        guihooks.trigger('CrashSurvivalRatingReset', {})
    end
end

-- ============================================================================
-- SETTINGS CHANGED CALLBACK
-- ============================================================================
local function onSettingsChanged()
    -- Request settings from GE extension
    -- For now, we use defaults - settings sync happens via GE extension callback
    print("crashSurvivalRating: Settings changed notification received")
end

-- ============================================================================
-- INITIALIZATION
-- ============================================================================
local function onInit()
    onReset()
    updateInterval = 1 / settings.updateFrequency

    print("====================================")
    print("crashSurvivalRating: Vehicle extension loaded")
    print("  Update rate: " .. settings.updateFrequency .. " Hz")
    print("  Impact threshold: " .. settings.impactThreshold .. " G")
    print("====================================")
end

-- ============================================================================
-- PUBLIC INTERFACE
-- ============================================================================
M.onUpdate = onUpdate
M.onReset = onReset
M.onInit = onInit
M.onSettingsChanged = onSettingsChanged

-- Expose current data for debugging
M.getData = function() return dataPacket end
M.getBodyParts = function() return bodyPartInjuries end

return M
