-- ============================================================================
-- Crash Survival Rating Pro v3.0 - Settings Module
-- Copyright (c) 2025 minMAX420 - All Rights Reserved
--
-- Licensed under CC BY-NC-SA 4.0 (Creative Commons Attribution-NonCommercial-ShareAlike)
-- https://creativecommons.org/licenses/by-nc-sa/4.0/
--
-- PERSONAL USE ONLY - Commercial use is strictly prohibited.
-- See LICENSE file for full terms.
-- ============================================================================
--
-- Settings Module - Handles user configuration and persistence

local M = {}

-- Default settings
local defaults = {
    -- Units
    useMetric = false,  -- false = imperial (MPH), true = metric (KPH)

    -- Update frequency
    updateFrequency = 20,  -- Hz (updates per second)

    -- G-force thresholds for impact detection
    impactThreshold = 5.0,  -- G-forces above this trigger impact count
    impactCooldown = 0.2,   -- Seconds between impact detections

    -- Injury level thresholds (total G-force)
    thresholds = {
        minor = 4.0,      -- Minor discomfort
        moderate = 8.0,   -- Moderate injury likely
        severe = 15.0,    -- Severe injury likely
        critical = 25.0,  -- Critical injury likely
        fatal = 40.0      -- Fatal injury likely
    },

    -- Duration thresholds (seconds at high G)
    highGThreshold = 4.0,     -- G-force considered "high"
    highGDurationWarn = 0.5,  -- Warning after this duration at high G

    -- Sound settings
    soundEnabled = true,
    soundVolume = 0.7,  -- 0.0 to 1.0
    soundWarningThreshold = 'severe',  -- Play sound at this injury level

    -- Display options
    showVehicleStats = true,
    showGForceBreakdown = true,
    showImpactCounter = true,
    showBodyPartIndicators = true,

    -- Advanced
    smoothingFactor = 0.3,  -- G-force smoothing (0 = no smoothing, 1 = max smoothing)
    debugMode = false
}

-- Current settings (loaded from storage or defaults)
local currentSettings = {}

-- Settings key for BeamNG storage
local SETTINGS_KEY = 'crashSurvivalRating_settings'

-- Deep copy a table
local function deepCopy(orig)
    local copy
    if type(orig) == 'table' then
        copy = {}
        for k, v in pairs(orig) do
            copy[k] = deepCopy(v)
        end
    else
        copy = orig
    end
    return copy
end

-- Merge user settings with defaults (user settings take precedence)
local function mergeWithDefaults(userSettings)
    local merged = deepCopy(defaults)
    if userSettings then
        for k, v in pairs(userSettings) do
            if type(v) == 'table' and type(merged[k]) == 'table' then
                for k2, v2 in pairs(v) do
                    merged[k][k2] = v2
                end
            else
                merged[k] = v
            end
        end
    end
    return merged
end

-- Load settings from BeamNG storage
local function loadSettings()
    local saved = nil

    -- Try to load from BeamNG settings system
    if settings and settings.getValue then
        saved = settings.getValue(SETTINGS_KEY)
    end

    -- Merge with defaults
    currentSettings = mergeWithDefaults(saved)

    print("crashSurvivalRating: Settings loaded")
    if currentSettings.debugMode then
        print("crashSurvivalRating: Debug mode enabled")
    end

    return currentSettings
end

-- Save settings to BeamNG storage
local function saveSettings()
    if settings and settings.setValue then
        settings.setValue(SETTINGS_KEY, currentSettings)
        print("crashSurvivalRating: Settings saved")
    else
        print("crashSurvivalRating: Warning - Could not save settings (settings API not available)")
    end
end

-- Get a single setting value
local function get(key)
    -- Handle nested keys like "thresholds.minor"
    if key:find('%.') then
        local parts = {}
        for part in key:gmatch('[^%.]+') do
            table.insert(parts, part)
        end
        local value = currentSettings
        for _, part in ipairs(parts) do
            if type(value) == 'table' then
                value = value[part]
            else
                return nil
            end
        end
        return value
    end
    return currentSettings[key]
end

-- Set a single setting value
local function set(key, value)
    -- Handle nested keys
    if key:find('%.') then
        local parts = {}
        for part in key:gmatch('[^%.]+') do
            table.insert(parts, part)
        end
        local target = currentSettings
        for i = 1, #parts - 1 do
            if type(target[parts[i]]) ~= 'table' then
                target[parts[i]] = {}
            end
            target = target[parts[i]]
        end
        target[parts[#parts]] = value
    else
        currentSettings[key] = value
    end

    -- Notify UI of settings change
    if guihooks then
        guihooks.trigger('CrashSurvivalRatingSettingsChanged', {
            key = key,
            value = value,
            allSettings = currentSettings
        })
    end
end

-- Get all settings
local function getAll()
    return deepCopy(currentSettings)
end

-- Reset to defaults
local function resetToDefaults()
    currentSettings = deepCopy(defaults)
    saveSettings()

    if guihooks then
        guihooks.trigger('CrashSurvivalRatingSettingsReset', currentSettings)
    end

    print("crashSurvivalRating: Settings reset to defaults")
end

-- Get defaults (for UI to show default values)
local function getDefaults()
    return deepCopy(defaults)
end

-- Validate a setting value
local function validate(key, value)
    -- Validation rules
    local validators = {
        updateFrequency = function(v) return type(v) == 'number' and v >= 1 and v <= 60 end,
        impactThreshold = function(v) return type(v) == 'number' and v >= 1 and v <= 50 end,
        impactCooldown = function(v) return type(v) == 'number' and v >= 0.1 and v <= 2.0 end,
        highGThreshold = function(v) return type(v) == 'number' and v >= 1 and v <= 20 end,
        highGDurationWarn = function(v) return type(v) == 'number' and v >= 0.1 and v <= 5.0 end,
        soundVolume = function(v) return type(v) == 'number' and v >= 0 and v <= 1 end,
        smoothingFactor = function(v) return type(v) == 'number' and v >= 0 and v <= 1 end,
        ['thresholds.minor'] = function(v) return type(v) == 'number' and v >= 1 and v <= 100 end,
        ['thresholds.moderate'] = function(v) return type(v) == 'number' and v >= 1 and v <= 100 end,
        ['thresholds.severe'] = function(v) return type(v) == 'number' and v >= 1 and v <= 100 end,
        ['thresholds.critical'] = function(v) return type(v) == 'number' and v >= 1 and v <= 100 end,
        ['thresholds.fatal'] = function(v) return type(v) == 'number' and v >= 1 and v <= 100 end
    }

    local validator = validators[key]
    if validator then
        return validator(value)
    end

    -- Boolean settings
    local booleanSettings = {
        'useMetric', 'soundEnabled', 'showVehicleStats', 'showGForceBreakdown',
        'showImpactCounter', 'showBodyPartIndicators', 'debugMode'
    }
    for _, boolKey in ipairs(booleanSettings) do
        if key == boolKey then
            return type(value) == 'boolean'
        end
    end

    return true  -- Allow unknown settings
end

-- Set a setting with validation
local function setValidated(key, value)
    if validate(key, value) then
        set(key, value)
        return true
    else
        print("crashSurvivalRating: Invalid value for setting '" .. key .. "': " .. tostring(value))
        return false
    end
end

-- Initialize on extension load
local function onExtensionLoaded()
    loadSettings()
end

-- Export functions
M.onExtensionLoaded = onExtensionLoaded
M.loadSettings = loadSettings
M.saveSettings = saveSettings
M.get = get
M.set = set
M.setValidated = setValidated
M.getAll = getAll
M.getDefaults = getDefaults
M.resetToDefaults = resetToDefaults
M.validate = validate

return M
