-- ============================================================================
-- Crash Survival Rating Pro v3.0 - GE Extension
-- Copyright (c) 2025 minMAX420 - All Rights Reserved
--
-- Licensed under CC BY-NC-SA 4.0 (Creative Commons Attribution-NonCommercial-ShareAlike)
-- https://creativecommons.org/licenses/by-nc-sa/4.0/
--
-- PERSONAL USE ONLY - Commercial use is strictly prohibited.
-- See LICENSE file for full terms.
-- ============================================================================

local M = {}

-- Track the current active vehicle for state management
local activeVehicleId = nil

-- Settings module reference
local settingsModule = nil

-- Called when extension loads
local function onExtensionLoaded()
    print("crashSurvivalRating: GE Extension loaded")

    -- Load the settings module
    settingsModule = require('crashSurvivalRating/settings')
    if settingsModule and settingsModule.onExtensionLoaded then
        settingsModule.onExtensionLoaded()
    end

    -- Send initial settings to UI
    if guihooks and settingsModule then
        guihooks.trigger('CrashSurvivalRatingSettingsLoaded', settingsModule.getAll())
    end
end

-- Called when extension unloads - cleanup
local function onExtensionUnloaded()
    print("crashSurvivalRating: GE Extension unloaded")
    activeVehicleId = nil

    -- Notify UI of cleanup
    if guihooks then
        guihooks.trigger('CrashSurvivalRatingUnloaded', {})
    end
end

-- Helper to safely get vehicle by ID
local function safeGetVehicle(vehId)
    if not be then
        print("crashSurvivalRating: Warning - BeamEngine (be) not available")
        return nil
    end
    return be:getObjectByID(vehId)
end

-- Called when a vehicle spawns
local function onVehicleSpawned(vehId)
    print("crashSurvivalRating: Vehicle spawned - " .. tostring(vehId))

    -- Load the vehicle extension into the spawned vehicle
    local veh = safeGetVehicle(vehId)
    if veh then
        veh:queueLuaCommand("extensions.load('crashSurvivalRating')")
        print("crashSurvivalRating: Loading vehicle extension into vehicle " .. tostring(vehId))
        activeVehicleId = vehId
    else
        print("crashSurvivalRating: Warning - Could not get vehicle object for ID " .. tostring(vehId))
    end

    -- Notify UI
    if guihooks then
        guihooks.trigger('VehicleSpawned', {vehicleId = vehId})
    end
end

-- Called when the player switches to a different vehicle
local function onVehicleSwitched(oldVehId, newVehId, player)
    print("crashSurvivalRating: Vehicle switched from " .. tostring(oldVehId) .. " to " .. tostring(newVehId))

    -- Load extension into new vehicle if not already loaded
    local veh = safeGetVehicle(newVehId)
    if veh then
        veh:queueLuaCommand("extensions.load('crashSurvivalRating')")
        activeVehicleId = newVehId
    end

    -- Notify UI of vehicle switch so it can reset display state
    if guihooks then
        guihooks.trigger('CrashSurvivalRatingVehicleSwitched', {
            oldVehicleId = oldVehId,
            newVehicleId = newVehId
        })
    end
end

-- Called when a vehicle is reset
local function onVehicleResetted(vehId)
    print("crashSurvivalRating: Vehicle reset - " .. tostring(vehId))

    -- Tell the vehicle extension to reset its state
    local veh = safeGetVehicle(vehId)
    if veh then
        veh:queueLuaCommand("if extensions.crashSurvivalRating then extensions.crashSurvivalRating.onReset() end")
    end

    -- Notify UI
    if guihooks then
        guihooks.trigger('VehicleReset', {vehicleId = vehId})
    end
end

-- Called when a vehicle is destroyed
local function onVehicleDestroyed(vehId)
    print("crashSurvivalRating: Vehicle destroyed - " .. tostring(vehId))

    -- Clear active vehicle if it was the destroyed one
    if activeVehicleId == vehId then
        activeVehicleId = nil
    end

    -- Notify UI
    if guihooks then
        guihooks.trigger('CrashSurvivalRatingVehicleDestroyed', {vehicleId = vehId})
    end
end

-- Get the current active vehicle ID
local function getActiveVehicleId()
    return activeVehicleId
end

-- Settings API functions (callable from UI via guihooks)

-- Get a single setting
local function getSetting(key)
    if settingsModule then
        return settingsModule.get(key)
    end
    return nil
end

-- Set a single setting (with validation)
local function setSetting(key, value)
    if settingsModule then
        local success = settingsModule.setValidated(key, value)
        if success then
            -- Propagate to vehicle extension if loaded
            local veh = safeGetVehicle(activeVehicleId)
            if veh then
                veh:queueLuaCommand("if extensions.crashSurvivalRating and extensions.crashSurvivalRating.onSettingsChanged then extensions.crashSurvivalRating.onSettingsChanged() end")
            end
        end
        return success
    end
    return false
end

-- Get all settings
local function getAllSettings()
    if settingsModule then
        return settingsModule.getAll()
    end
    return {}
end

-- Get default settings
local function getDefaultSettings()
    if settingsModule then
        return settingsModule.getDefaults()
    end
    return {}
end

-- Reset settings to defaults
local function resetSettings()
    if settingsModule then
        settingsModule.resetToDefaults()
        -- Propagate to vehicle extension
        local veh = safeGetVehicle(activeVehicleId)
        if veh then
            veh:queueLuaCommand("if extensions.crashSurvivalRating and extensions.crashSurvivalRating.onSettingsChanged then extensions.crashSurvivalRating.onSettingsChanged() end")
        end
    end
end

-- Save settings to persistent storage
local function saveSettings()
    if settingsModule then
        settingsModule.saveSettings()
    end
end

-- Export functions
M.onExtensionLoaded = onExtensionLoaded
M.onExtensionUnloaded = onExtensionUnloaded
M.onVehicleSpawned = onVehicleSpawned
M.onVehicleSwitched = onVehicleSwitched
M.onVehicleResetted = onVehicleResetted
M.onVehicleDestroyed = onVehicleDestroyed
M.getActiveVehicleId = getActiveVehicleId

-- Settings exports
M.getSetting = getSetting
M.setSetting = setSetting
M.getAllSettings = getAllSettings
M.getDefaultSettings = getDefaultSettings
M.resetSettings = resetSettings
M.saveSettings = saveSettings

return M
